package com.instabug.terminations.model

import android.content.Context
import android.net.Uri
import androidx.annotation.WorkerThread
import com.instabug.commons.AttachmentsHolder
import com.instabug.commons.BasicAttachmentsHolder
import com.instabug.commons.caching.DiskHelper
import com.instabug.commons.caching.DiskHelper.getReproScreenshotsZipPath
import com.instabug.commons.models.Incident
import com.instabug.commons.models.Incident.Type
import com.instabug.commons.models.IncidentMetadata
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation
import com.instabug.library.model.Attachment
import com.instabug.library.model.State
import java.io.File

private const val APP_TERMINATION_STATE = "app_termination_state"

data class Termination(
    override val metadata: IncidentMetadata,
    val id: Long
) : Incident, AttachmentsHolder by BasicAttachmentsHolder() {

    @TerminationState
    var incidentState: Int = TerminationState.READY_TO_BE_SENT
    var temporaryServerToken: String? = null
    var state: State? = null
    var stateUri: Uri? = null
    var sessionId: String? = null

    override val type: Type = Type.Termination
    override fun getSavingDirOnDisk(ctx: Context): File =
        DiskHelper.getIncidentSavingDirectory(ctx, type.name, id.toString())

    fun readStateFile(context: Context) {
        state = State.getState(context, stateUri)
    }

    fun clearStateFile() {
        state = null
    }

    fun toNextState(): Int = ++incidentState

    object Factory {
        operator fun invoke(
            ctx: Context?,
            id: Long,
            sessionId: String,
            state: State?,
            screenshotsDir: File?,
            metadata: IncidentMetadata = IncidentMetadata.Factory.create()
        ) = Termination(metadata, id).apply {
            state?.setAppStatusToForeground()
            stateUri = ctx?.let { state?.persist(it, getSavingDirOnDisk(it)) }
            screenshotsDir?.also { dir -> ctx?.let { addReproScreenshotsAttachment(this, it, dir) } }
            this.sessionId = sessionId
        }

        operator fun invoke(
            id: Long,
            metadata: IncidentMetadata,
            creator: Termination.() -> Unit
        ) = Termination(metadata, id).apply(creator)

        private fun State.persist(context: Context, savingDir: File): Uri =
            DiskHelper.getIncidentStateFile(savingDir, APP_TERMINATION_STATE).let {
                DiskUtils.with(context)
                    .writeOperation(WriteStateToFileDiskOperation(it, toJson()))
                    .execute()
            }

        @WorkerThread
        private fun addReproScreenshotsAttachment(
            termination: Termination,
            context: Context,
            screenshotsDir: File
        ) {
            val (zipPath, isEncrypted) = getReproScreenshotsZipPath(
                context,
                termination.id.toString(),
                termination.getSavingDirOnDisk(context),
                screenshotsDir
            )
            zipPath?.let { path ->
                termination.addAttachment(
                    Uri.parse(path),
                    Attachment.Type.VISUAL_USER_STEPS,
                    isEncrypted
                )
            }
        }
    }
}
