package com.instabug.terminations.cache

import androidx.annotation.WorkerThread
import com.instabug.commons.caching.FileCacheDirectory
import com.instabug.commons.caching.SessionCacheDirectory
import com.instabug.commons.snapshot.rename
import java.io.File

class TerminationsCacheDir(private val parentDir: SessionCacheDirectory) : FileCacheDirectory {
    override val fileDirectory: File?
        @WorkerThread get() = parentDir.currentSessionDirectory?.let(::getTerminationsDir)

    companion object {
        private const val TERMINATIONS_DIR_NAME = "trm"
        private const val TRM_SNAPSHOT_FILE_NAME = "trm-snapshot"
        private const val OLD_FILE_SUFFIX = "-old"
        const val BASELINE_FILE_SUFFIX = "-bl"
        const val DETECTED_FILE_SUFFIX = "-osd"
        const val VALIDATED_FILE_SUFFIX = "-vld"
        const val MIGRATED_FILE_SUFFIX = "-mig"
        const val GROUND_STATE_FG = "-fg"
        const val GROUND_STATE_BG = "-bg"

        fun getTerminationsDir(sessionDir: File): File =
            sessionDir.run { File("$absolutePath${File.separator}$TERMINATIONS_DIR_NAME") }

        fun getBaselineFile(sessionDir: File, timestamp: Long): File =
            getTerminationsDir(sessionDir).run { File("$absolutePath${File.separator}$timestamp$BASELINE_FILE_SUFFIX") }

        fun getBaselineFile(sessionDir: File): File? =
            getTerminationsDir(sessionDir).takeIf { it.exists() }
                ?.listFiles { file -> file.nameWithoutExtension.endsWith(BASELINE_FILE_SUFFIX) }
                ?.firstOrNull()

        fun getDetectedFile(sessionDir: File, timestamp: Long): File =
            getTerminationsDir(sessionDir).run { File("$absolutePath${File.separator}$timestamp$DETECTED_FILE_SUFFIX") }

        fun getDetectedFile(sessionDir: File): File? =
            getTerminationsDir(sessionDir).takeIf { it.exists() }
                ?.listFiles { file -> file.nameWithoutExtension.endsWith(DETECTED_FILE_SUFFIX) }
                ?.firstOrNull()

        fun getValidatedFile(sessionDir: File, timestamp: Long): File =
            getTerminationsDir(sessionDir).run { File("$absolutePath${File.separator}$timestamp$VALIDATED_FILE_SUFFIX") }

        fun getValidatedFile(sessionDir: File): File? =
            getTerminationsDir(sessionDir).takeIf { it.exists() }
                ?.listFiles { file -> file.nameWithoutExtension.endsWith(VALIDATED_FILE_SUFFIX) }
                ?.firstOrNull()

        fun markBaselineFileAsDetected(baselineFile: File, groundState: String, timestamp: Long) {
            baselineFile.rename("$timestamp$groundState$DETECTED_FILE_SUFFIX")
        }

        fun markDetectedFileAsValidated(detectedFile: File, stateSuffix: String): Unit =
            detectedFile.run { rename("${name.removeSuffix("$stateSuffix$DETECTED_FILE_SUFFIX")}$VALIDATED_FILE_SUFFIX") }

        fun markDetectionFileAsMigrated(detectionFile: File, suffix: String) {
            detectionFile.run { rename("${name.removeSuffix(suffix)}$MIGRATED_FILE_SUFFIX") }
        }

        fun getTerminationSnapshotFile(terminationDir: File): File =
            terminationDir.run { File("$absolutePath${File.separator}$TRM_SNAPSHOT_FILE_NAME.json") }

        fun getOldTerminationSnapshotFile(terminationDir: File): File? =
            terminationDir.run { File(terminationDir,"$TRM_SNAPSHOT_FILE_NAME$OLD_FILE_SUFFIX.json") }

        fun markSnapshotFileAsOld(snapshotFile: File): Unit =
            snapshotFile.run { rename("$nameWithoutExtension$OLD_FILE_SUFFIX.json") }

        fun markTerminationSnapshotFileAs(sessionDir: File, suffix: String) {
            getTerminationsDir(sessionDir).let { trmDir ->
                getTerminationSnapshotFile(trmDir).takeIf { it.exists() }
                    ?: getOldTerminationSnapshotFile(trmDir).takeIf { it?.exists() == true }
            }?.run { rename("$nameWithoutExtension$suffix.json") }
        }
    }
}

