package com.instabug.terminations

import android.app.ActivityManager
import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.commons.Detection
import com.instabug.commons.IncidentDetectorsListener
import com.instabug.commons.IncidentDetectorsListenersRegistry
import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.snapshot.AbstractCaptor
import com.instabug.commons.snapshot.Captor
import com.instabug.commons.snapshot.CaptorConfigurations
import com.instabug.commons.snapshot.ifNotExists
import com.instabug.commons.snapshot.readJsonObject
import com.instabug.commons.snapshot.writeString
import com.instabug.commons.utils.isAtLeastRunningR
import com.instabug.commons.utils.isProcessInForeground
import com.instabug.library.core.InstabugCore
import com.instabug.library.model.v3Session.IBGSessionMapper.getCompositeSessionId
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.extenstions.optNullableString
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getOldTerminationSnapshotFile
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getTerminationSnapshotFile
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.markSnapshotFileAsOld
import com.instabug.terminations.di.ServiceLocator
import org.json.JSONArray
import org.json.JSONObject
import java.io.File
import java.io.Serializable
import java.util.concurrent.ScheduledExecutorService

const val FOREGROUND_TIMELINE: String = "FOREGROUND_TIMELINE"
const val SESSION_COMPOSITE_ID: String = "SESSION_COMPOSITE_ID"

/**
 * A generic contract for snapshots to be saved using [AbstractTerminationSnapshotCaptor]
 */
interface TerminationSnapshot : Serializable {
    /**
     * A series of ground states to act as a timeline
     */
    val foregroundTimeline: List<Boolean>

    val sessionCompositeId: String?

    companion object {
        fun getCompositeSessionId(oldSnapshot: TerminationSnapshot?): String? =
            oldSnapshot?.sessionCompositeId ?: run {
                val runningInMemorySession = InstabugCore.getRunningV3Session()
                val appToken = SettingsManager.getInstance().appToken
                runningInMemorySession?.let { runningSession ->
                    appToken?.let { token -> runningSession.getCompositeSessionId(token) }
                }
            }
    }

    fun toJson(): JSONObject? {
        return runCatching {
            JSONObject().apply {
                val jsonArray = JSONArray().apply {
                    for (isForeground in foregroundTimeline) {
                        put(isForeground)
                    }
                }
                put(FOREGROUND_TIMELINE, jsonArray)
                put(SESSION_COMPOSITE_ID, sessionCompositeId ?: JSONObject.NULL)
            }
        }.getOrReportError(
            null,
            "error while serialized ${this.javaClass} to JsonObject"
        )
    }
}

/**
 * The snapshot to be saved in session directory for devices running Android R or above (API 30+).
 * Only interested in application ground state.
 */
@RequiresApi(Build.VERSION_CODES.R)
data class PostAndroidRTerminationSnapshot(
    override val foregroundTimeline: List<Boolean>,
    override val sessionCompositeId: String?
) : TerminationSnapshot {
    object Factory {
        /**
         * Used to create an object of the snapshot using [Context] argument.
         * Determines the ground state of the app using [ActivityManager] API
         * @see ActivityManager.getRunningAppProcesses
         * @param ctx Application context
         */
        operator fun invoke(
            ctx: Context,
            oldSnapshot: PostAndroidRTerminationSnapshot?
        ): PostAndroidRTerminationSnapshot =
            PostAndroidRTerminationSnapshot(
                getUpdatedTimeline(ctx.isProcessInForeground, oldSnapshot),
                TerminationSnapshot.getCompositeSessionId(oldSnapshot)
            )

        private fun getUpdatedTimeline(
            newState: Boolean,
            oldSnapshot: PostAndroidRTerminationSnapshot?
        ): List<Boolean> = oldSnapshot?.foregroundTimeline?.toMutableList()
            ?.apply { add(newState) }
            ?.run { if (size > 10) subList(1, size) else this }
            ?.toList() ?: listOf(newState)
    }

    companion object {
        @JvmStatic
        fun fromJson(jsonObject: JSONObject?): PostAndroidRTerminationSnapshot? {
            return kotlin.runCatching {
                jsonObject ?: return null

                val jsonArray = jsonObject.getJSONArray(FOREGROUND_TIMELINE)

                val foregroundTimeline = MutableList(jsonArray.length()) { index ->
                    jsonArray.getBoolean(index)
                }

                val sessionCompositeId = jsonObject.optNullableString(SESSION_COMPOSITE_ID)

                PostAndroidRTerminationSnapshot(
                    foregroundTimeline = foregroundTimeline,
                    sessionCompositeId = sessionCompositeId
                )
            }.getOrReportError(
                null,
                "Error while parsing the PostAndroidRTerminationSnapshot."
            )
        }
    }
}

const val TIMESTAMP = "TIMESTAMP"
const val IS_IN_ANR = "IS_IN_ANR"
const val HAS_CRASHED = "HAS_CRASHED"

/**
 * The snapshot to be saved in session directory for devices running Android below R (API 29-).
 * Saves the timestamp of capturing, app ground state, whether the app is in an ANR and whether
 * the app has crashed.
 */
data class PreAndroidRTerminationSnapshot(
    val timestamp: Long,
    override val foregroundTimeline: List<Boolean>,
    override val sessionCompositeId: String?,
    val isInAnr: Boolean,
    val hasCrashed: Boolean
) : TerminationSnapshot {
    object Factory {
        /**
         * Used to construct a new snapshot.
         * If an old snapshot is provided without any detections [Detection], then old snapshot shall
         * be copied updating only the timestamp and the ground state.
         * In case of a detection, the new snapshot shall reflect that detection.
         * @param ctx Application context to get ground state.
         * @param oldSnapshot The previous snapshot taken to be replicated with changed fields update only.
         * @param detection A [Detection] reported by either the ANR detector or the crash detector.
         */
        operator fun invoke(
            ctx: Context,
            oldSnapshot: PreAndroidRTerminationSnapshot? = null,
            @Detection detection: String? = null
        ): PreAndroidRTerminationSnapshot = PreAndroidRTerminationSnapshot(
            timestamp = System.currentTimeMillis(),
            foregroundTimeline = getUpdatedTimeline(ctx.isProcessInForeground, oldSnapshot),
            TerminationSnapshot.getCompositeSessionId(oldSnapshot),
            isInAnr = detection.isInAnr(oldSnapshot?.isInAnr ?: false),
            hasCrashed = detection.hasCrashed(oldSnapshot?.hasCrashed ?: false)
        )

        private fun String?.isInAnr(oldState: Boolean) = when (this) {
            Detection.Anr -> true
            Detection.AnrRecovery -> false
            else -> oldState
        }

        private fun String?.hasCrashed(oldState: Boolean) = when (this) {
            Detection.Crash -> true
            else -> oldState
        }

        private fun getUpdatedTimeline(
            newState: Boolean,
            oldSnapshot: PreAndroidRTerminationSnapshot?
        ): List<Boolean> = oldSnapshot?.foregroundTimeline?.toMutableList()
            ?.apply { add(newState) }
            ?.run { if (size > 10) subList(1, size) else this }
            ?.toList() ?: listOf(newState)
    }

    override fun toJson(): JSONObject? {
        return runCatching {
            super.toJson()?.apply {
                put(TIMESTAMP, timestamp)
                put(IS_IN_ANR, isInAnr)
                put(HAS_CRASHED, hasCrashed)
            }
        }.getOrReportError(
            null,
            "error while serialized PreAndroidRTerminationSnapshot to JsonObject"
        )
    }

    companion object {
        @JvmStatic
        fun fromJson(jsonObject: JSONObject?): PreAndroidRTerminationSnapshot? {
            return kotlin.runCatching {
                jsonObject ?: return null
                val jsonArray = jsonObject.getJSONArray(FOREGROUND_TIMELINE)

                val foregroundTimeline = MutableList(jsonArray.length()) { index ->
                    jsonArray.getBoolean(index)
                }

                val sessionCompositeId = jsonObject.optNullableString(SESSION_COMPOSITE_ID)
                val timestamp = jsonObject.getLong(TIMESTAMP)
                val isInANR = jsonObject.getBoolean(IS_IN_ANR)
                val hasCrashed = jsonObject.getBoolean(HAS_CRASHED)
                PreAndroidRTerminationSnapshot(
                    timestamp,
                    foregroundTimeline,
                    sessionCompositeId,
                    isInANR,
                    hasCrashed
                )
            }.getOrReportError(
                null,
                "Error while parsing the PreAndroidRTerminationSnapshot."
            )
        }
    }
}

abstract class AbstractTerminationSnapshotCaptor<T : TerminationSnapshot>(
    protected val configurations: CaptorConfigurations
) : AbstractCaptor(configurations.scheduler) {

    final override val id: Int
        get() = ID
    final override val capturingPeriod: Long
        get() = STD_CAPTOR_PERIOD

    final override fun capture() {
        if (Thread.currentThread().isInterrupted) return
        internalCapture(this::getSnapshot)
    }

    abstract fun getSnapshot(ctx: Context, oldSnapshotAsJsonObject: JSONObject? = null): T

    protected fun internalCapture(snapshotGetter: (Context, JSONObject?) -> T) {
        configurations.savingDirectory?.let { savingDir ->
            //Mark existing snapshot with -old/.
            getTerminationSnapshotFile(savingDir)
                .takeIf { it.exists() }
                ?.let { snapshotFile -> markSnapshotFileAsOld(snapshotFile) }
            configurations.ctx?.let { ctx ->
                savingDir.ifNotExists { mkdirs() }
                //Get the old snapshot to be passed for replication.
                val oldSnapshot = getOldTerminationSnapshotFile(savingDir)
                    .takeIf { it?.exists() == true }
                    ?.let(File::readJsonObject)

                //Save the new snapshot.
                getTerminationSnapshotFile(savingDir)
                    .writeString(
                        snapshotGetter(ctx, oldSnapshot).toJson().toString()
                    )
            }
            //Delete old snapshot.
            getOldTerminationSnapshotFile(savingDir).takeIf { it?.exists() == true }?.delete()
        }
    }

    object Factory {
        operator fun invoke(
            ctxGetter: () -> Context? = ServiceLocator::appCtx,
            savingDirectoryGetter: () -> File? = ServiceLocator.terminationsCacheDir::fileDirectory,
            scheduler: ScheduledExecutorService = ServiceLocator.getCoreScheduler()
        ): Captor = CaptorConfigurations(ctxGetter, savingDirectoryGetter, scheduler)
            .let { configurations ->
                if (isAtLeastRunningR) PostAndroidRTerminationSnapshotCaptor(configurations)
                else PreAndroidRTerminationSnapshotCaptor(
                    configurations,
                    ServiceLocator.detectorsListenersRegistry
                )
            }
    }

    companion object {
        const val ID = 0x002
        private const val STD_CAPTOR_PERIOD = 2L
    }
}

@RequiresApi(Build.VERSION_CODES.R)
class PostAndroidRTerminationSnapshotCaptor(
    configurations: CaptorConfigurations
) : AbstractTerminationSnapshotCaptor<PostAndroidRTerminationSnapshot>(configurations) {

    override fun getSnapshot(
        ctx: Context,
        oldSnapshotAsJsonObject: JSONObject?
    ): PostAndroidRTerminationSnapshot {
        val oldSnapshot = PostAndroidRTerminationSnapshot.fromJson(oldSnapshotAsJsonObject)
        return PostAndroidRTerminationSnapshot.Factory(ctx, oldSnapshot)
    }

    override fun onStart() {
        "Starting termination snapshot captor".logVerbose()
    }

    override fun onShutdown() {
        "Shutting down termination snapshot captor".logVerbose()
    }
}

class PreAndroidRTerminationSnapshotCaptor(
    configurations: CaptorConfigurations,
    private val listenersRegistry: IncidentDetectorsListenersRegistry
) : AbstractTerminationSnapshotCaptor<PreAndroidRTerminationSnapshot>(configurations),
    IncidentDetectorsListener {

    override fun getSnapshot(
        ctx: Context,
        oldSnapshotAsJsonObject: JSONObject?
    ): PreAndroidRTerminationSnapshot {
        val oldSnapshot = PreAndroidRTerminationSnapshot.fromJson(oldSnapshotAsJsonObject)
        return PreAndroidRTerminationSnapshot.Factory(ctx, oldSnapshot)
    }

    override fun onStart() {
        listenersRegistry.register(this)
        "Starting termination snapshot captor".logVerbose()
    }

    override fun onShutdown() {
        listenersRegistry.unregister(this)
        "Shutting down termination snapshot captor".logVerbose()
    }

    override fun onDetection(@Detection detection: String) = synchronized(this) {
        if (isShutdown) return
        "Trm snapshot captor received detection: $detection".logVerbose()
        internalPause()
        val snapshotGetter = { ctx: Context, oldSnapshot: Any? ->
            (oldSnapshot as? PreAndroidRTerminationSnapshot).let { old ->
                PreAndroidRTerminationSnapshot.Factory(ctx, old, detection)
            }
        }
        scheduler.execute { internalCapture(snapshotGetter); internalStart(capturingPeriod) }
    }
}
