package com.instabug.fatalhangs

import android.os.Debug
import android.os.Handler
import android.os.Looper
import androidx.annotation.WorkerThread
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.logging.runOrLogError
import com.instabug.commons.models.IncidentMetadata
import com.instabug.commons.session.SessionIncident.ValidationStatus
import com.instabug.commons.threading.CrashDetailsParser
import com.instabug.fatalhangs.di.FatalHangsServiceLocator
import com.instabug.fatalhangs.model.FatalHang
import com.instabug.library.Instabug
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicLong

private const val DEFAULT_CHECK_INTERVAL = 500L

/**
 * Designed to detect jams in main thread. Does so as follows
 * 1. Posts a runnable to be executed on the main thread (ticker).
 * 2. Halts the thread execution for 500ms (Check interval).
 * 3. Check if the waiting time for the ticker to be executed reached the defined sensitivity.
 * 4. If waiting time for the ticker to execute < sensitivity go back to 1.
 * 5. If waiting time for the ticker to execute >= sensitivity report a fatal hang and go back to 1.
 */
class IBGFatalHangDetector(
    private val callback: (FatalHang) -> Unit,
    targetThreadLooper: Looper = Looper.getMainLooper()
) : Thread() {

    private var isThreadInterrupted = false

    private val sensitivity
        get() = FatalHangsServiceLocator.fatalHangsConfigurationProvider.fatalHangsSensitivity

    private val tick by lazy { AtomicLong(0) }

    private val reported by lazy { AtomicBoolean(false) }

    private val ticker: () -> Unit = { tick.set(0); reported.set(false) }

    private val targetThreadHandler by lazy { Handler(targetThreadLooper) }

    override fun interrupt() {
        isThreadInterrupted = true
        super.interrupt()
    }

    override fun run() {
        name = "Instabug Fatal Hang detector thread"
        while (!isThreadInterrupted)
            runCatching { runHangCheck() }
                .runOrLogError("Error running fatal hangs check")
    }

    private fun runHangCheck() {
        // Add 500ms to current tick value and post a new ticker if previous tick value was 0
        tick.getAndAdd(DEFAULT_CHECK_INTERVAL).takeIf { it == 0L }?.run { postTicker() }

        // Halts the thread for 500ms
        runCatching { sleep(DEFAULT_CHECK_INTERVAL) }

        // If tick value (total waiting time) didn't reach defined sensitivity
        // or the current jam has been already reported, don't proceed in reporting!
        if (tick.get() < sensitivity || reported.get()) return
        // If the debugger is attached or going to be, don't proceed in reporting!
        if (Debug.isDebuggerConnected() || Debug.waitingForDebugger()) return
        "Fatal hang detected".logVerbose()
        runCatching { reportFatalHang() }
        // Mark the current jam as reported not till a new jam is happening
        // not to report the same hang twice
        reported.set(true)
    }

    private fun postTicker() {
        targetThreadHandler.post(ticker)
    }

    private fun reportFatalHang() {
        val detailsSnapshot = CrashDetailsParser(
            threadParsingStrategy = CrashDetailsParser.ThreadParsingStrategy.Main,
            errorParsingStrategy = CrashDetailsParser.ErrorParsingStrategy.Main()
        )
        with(FatalHangsServiceLocator) {
            getIOExecutor()?.execute { createAndLinkIncident(detailsSnapshot) }
        }
    }

    @WorkerThread
    fun createAndLinkIncident(detailsSnapshot: CrashDetailsParser) {
        runCatching {
            createIncident(detailsSnapshot)
                ?.also { incident ->
                    FatalHangsServiceLocator.sessionLinker.link(
                        incident,
                        ValidationStatus.VALIDATED
                    )
                }?.let(callback)
        }.runOrLogError ("Error creating Fatal Hang incident")
    }

    @WorkerThread
    private fun createIncident(detailsSnapshot: CrashDetailsParser) =
        FatalHang.Factory.createFatalHang(
            Instabug.getApplicationContext(),
            sensitivity,
            detailsSnapshot.crashDetails,
            detailsSnapshot.threadsDetails.toString(),
            IncidentMetadata.Factory.create()
        )
}
