package com.instabug.commons

import android.app.ActivityManager
import android.app.ActivityManager.RunningAppProcessInfo
import android.app.ApplicationExitInfo
import android.content.Context
import android.content.SharedPreferences
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.commons.OSExitInfoExtractor.Result
import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.preferences.PrefSpec
import com.instabug.commons.preferences.crashPref
import com.instabug.commons.utils.activityManager
import com.instabug.commons.utils.exitInfo
import java.io.InputStream

data class OSExitInfo(
    val internalReason: Int,
    val timestamp: Long,
    val importance: Int,
    val description: String?,
    val traceStream: () -> InputStream?
)

@RequiresApi(Build.VERSION_CODES.R)
fun OSExitInfo.isUserTermination(): Boolean =
    internalReason == ApplicationExitInfo.REASON_USER_REQUESTED

@RequiresApi(Build.VERSION_CODES.R)
fun OSExitInfo.isAnr(): Boolean =
    internalReason == ApplicationExitInfo.REASON_ANR

@RequiresApi(Build.VERSION_CODES.R)
fun OSExitInfo.isOfVisibleImportance(): Boolean =
    isExplicitlyInForeground() || importance == RunningAppProcessInfo.IMPORTANCE_FOREGROUND_SERVICE

@RequiresApi(Build.VERSION_CODES.R)
fun OSExitInfo.isExplicitlyInForeground(): Boolean =
    importance == RunningAppProcessInfo.IMPORTANCE_FOREGROUND

@RequiresApi(Build.VERSION_CODES.R)
fun OSExitInfo.safelyActOnTraceStream(action: (InputStream) -> Unit): Boolean =
    traceStream()?.use(action) != null

interface OSExitInfoExtractor {
    fun extract(ctx: Context, baselinePrefSpec: PrefSpec<Long>): Result
    fun extract(ctx: Context, extractionBaseline: Long): Result
    fun extractByCount(ctx: Context, count: Int): List<OSExitInfo>

    class Result(
        val oldBaseline: Long,
        val currentBaseline: Long,
        val infoList: List<OSExitInfo>
    )
}

@RequiresApi(Build.VERSION_CODES.R)
class DefaultOSExitInfoExtractor : OSExitInfoExtractor {
    /**
     * Extracts OS exit info with a baseline time for extraction.
     * Given a [PrefSpecs] of baseline time stored in [SharedPreferences], calling this method
     * automatically updates the stored to time to the new time of extraction (the time of invocation).
     * @param ctx A [Context] to be able to extract info from [ActivityManager]
     * @param baselinePrefSpec the [PrefSpecs] of baseline time stored in [SharedPreferences]
     * @return An instance of [OSExitInfoExtractor.Result] with an empty list of [OSExitInfo]s if
     * the extraction failed for any reason.
     */
    override fun extract(ctx: Context, baselinePrefSpec: PrefSpec<Long>): Result = run {
        var timeBaseline by crashPref(baselinePrefSpec)
        val currentTimeBaseline = timeBaseline
        timeBaseline = System.currentTimeMillis()
        extractInternally(ctx, currentTimeBaseline, timeBaseline)
    }

    /**
     * Extracts OS exit info with a baseline time for extraction.
     * @param ctx A [Context] to be able to extract info from [ActivityManager]
     * @param extractionBaseline The time stamp representing the baseline of extraction.
     * @return An instance of [OSExitInfoExtractor.Result] with an empty list of [OSExitInfo]s if
     * the extraction failed for any reason.
     */
    override fun extract(ctx: Context, extractionBaseline: Long): Result {
        val extractionTime = System.currentTimeMillis()
        return extractInternally(ctx, extractionBaseline, extractionTime)
    }

    override fun extractByCount(ctx: Context, count: Int): List<OSExitInfo> {
        return ctx
            .activityManager
            .getHistoricalProcessExitReasons(null, 0, count)
            .map { info -> toOSExitReason(info) }
    }

    private fun extractInternally(
        ctx: Context,
        extractionBaseline: Long,
        extractionTime: Long
    ): Result = runCatching {
        ctx.exitInfo.filter { info -> info.timestamp > extractionBaseline }
            .takeUnless { extractionBaseline < 0 }
            ?.map { info -> toOSExitReason(info) }
            .let { reasonsList -> reasonsList ?: emptyList() }
            .let { reasonsList -> Result(extractionBaseline, extractionTime, reasonsList) }
    }.getOrReportError(
        default = Result(extractionBaseline, extractionTime, emptyList()),
        message = "Couldn't extract OS exit info",
        withThrowable = false
    )

    private fun toOSExitReason(info: ApplicationExitInfo) =
        OSExitInfo(
            info.reason,
            info.timestamp,
            info.importance,
            info.description
        ) {
            info.traceInputStream
        }
}
