package com.instabug.terminations.cache

import android.content.Context
import android.database.Cursor
import android.net.Uri
import com.instabug.commons.models.IncidentMetadata
import com.instabug.crash.utils.deleteAttachment
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.AppTerminationEntry.*
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.terminations.Constants
import com.instabug.terminations.di.ServiceLocator
import com.instabug.terminations.model.Termination

class TerminationsCachingManagerImpl : TerminationsCachingManager {

    override fun insertAndTrim(context: Context, termination: Termination) {
        insert(termination)
        trim(context, ServiceLocator.terminationStoreLimit)
    }

    override fun update(termination: Termination): Int = kotlin.runCatching {
        InstabugSDKLogger.d(Constants.LOG_TAG, "DB->Updating termination ${termination.id}")
        IBGWhereArg(termination.id.toString(), true).let { arg ->
            IBGDbManager.getInstance().update(
                TABLE_NAME,
                termination.toContentValues(),
                "$COLUMN_ID = ?",
                listOf(arg)
            )
        }
    }.getOrReport(0, "Failed to update termination")

    override fun delete(context: Context, termination: Termination): Int = kotlin.runCatching {
        InstabugSDKLogger.d(Constants.LOG_TAG, "DB->Deleting termination ${termination.id}")
        termination.stateUri?.let { uri ->
            DiskUtils.with(context)
                .deleteOperation(DeleteUriDiskOperation(uri))
                .execute()
        }
        deleteAttachments(termination)
        IBGWhereArg(termination.id.toString(), true).let { arg ->
            IBGDbManager.getInstance().delete(
                TABLE_NAME,
                "$COLUMN_ID = ?",
                listOf(arg)
            )
        }
    }.getOrReport(0, "Failed to delete termination")

    override fun clear(context: Context) = kotlin.runCatching {
        trim(context, 0)
        Unit
    }.getOrReport(Unit, "Failed to clear terminations")

    override fun retrieve(context: Context): List<Termination> = kotlin.runCatching {
        InstabugSDKLogger.d(Constants.LOG_TAG, "DB->Retrieving all terminations")
        IBGDbManager.getInstance().query()?.use { cursor ->
            if (cursor.moveToFirst()) {
                cursor.toTerminationsList(context, false)
            } else {
                emptyList()
            }
        } ?: emptyList()
    }.getOrReport(emptyList(), "Failed to retrieve terminations")

    private fun insert(termination: Termination): Long = kotlin.runCatching {
        InstabugSDKLogger.d(Constants.LOG_TAG, "DB->Inserting termination ${termination.id}")
        insertAttachments(termination)
        IBGDbManager.getInstance()
            .insert(TABLE_NAME, null, termination.toContentValues())
    }.getOrReport(-1L, "Failed to insert termination")

    private fun trim(context: Context, limit: Int) = kotlin.runCatching {
        InstabugSDKLogger.d(Constants.LOG_TAG, "DB->Trimming terminations")
        IBGDbManager.getInstance().query()?.use { cursor ->
            if (cursor.count > limit) {
                val delta = cursor.count - limit
                cursor.moveToFirst()
                for (index in 0 until delta) {
                    delete(context, cursor.toTermination(context))
                    cursor.moveToNext()
                }
            }
        }
    }.getOrReport(Unit, "Failed to trim terminations")

    private fun Termination.toContentValues(): IBGContentValues = IBGContentValues().apply {
        put(COLUMN_ID, id, true)
        put(COLUMN_TERMINATION_STATE, incidentState, true)
        temporaryServerToken?.let { put(COLUMN_TEMPORARY_SERVER_TOKEN, it, true) }
        stateUri?.let { put(COLUMN_STATE, it.toString(), true) }
        metadata.uuid?.let { uuid -> put(COLUMN_UUID, uuid, true) }
    }

    private fun IBGDbManager.query(
        tableName: String = TABLE_NAME,
        selection: String? = null,
        args: List<IBGWhereArg>? = null,
        order: String? = null,
        limit: Int? = null
    ) = query(tableName, null, selection, args, null, null, order, limit?.toString())

    private fun Cursor.toTermination(context: Context, readStateFile: Boolean = false) =
        Termination.Factory(
            getLong(getColumnIndexOrThrow(COLUMN_ID)),
            IncidentMetadata.Factory.create(getString(getColumnIndexOrThrow(COLUMN_UUID)))
        ) {
            incidentState = getInt(getColumnIndexOrThrow(COLUMN_TERMINATION_STATE))
            temporaryServerToken = getString(getColumnIndexOrThrow(COLUMN_TEMPORARY_SERVER_TOKEN))
            stateUri = getString(getColumnIndexOrThrow(COLUMN_STATE))?.let { Uri.parse(it) }
            if (readStateFile) readStateFile(context)
            setAttachments(AttachmentsDbHelper.retrieve(id.toString()))
        }

    private fun Cursor.toTerminationsList(
        context: Context,
        readStateFile: Boolean = false
    ): List<Termination> {
        val terminations = mutableListOf<Termination>()
        do {
            terminations.add(toTermination(context, readStateFile))
        } while (moveToNext())
        return terminations
    }

    private fun <R> Result<R>.getOrReport(default: R, message: String) = getOrElse { throwable ->
        InstabugSDKLogger.e(Constants.LOG_TAG, message, throwable)
        InstabugCore.reportError(throwable, message)
        default
    }

    private fun insertAttachments(termination: Termination) {
        termination.attachments.forEach { attachment ->
            AttachmentsDbHelper.insert(attachment, termination.id.toString())
        }
    }

    private fun deleteAttachments(termination: Termination) {
        termination.attachments.forEach { attachment ->
            deleteAttachment(attachment, termination.id.toString())
        }
    }
}
