package com.instabug.early_crash.di

import android.app.Application
import android.content.Context
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.di.CommonsLocator
import com.instabug.crash.di.CrashesServiceLocator
import com.instabug.crash.settings.CrashSettings
import com.instabug.early_crash.caching.EarlyCrashCacheHandler
import com.instabug.early_crash.caching.IEarlyCrashCacheHandler
import com.instabug.early_crash.caching.ReportsDirectory
import com.instabug.early_crash.configurations.EarlyCrashesConfigHandler
import com.instabug.early_crash.configurations.EarlyCrashesConfigPersistence
import com.instabug.early_crash.configurations.EarlyCrashesConfigProvider
import com.instabug.early_crash.configurations.IEarlyCrashesConfigProvider
import com.instabug.early_crash.model.EarlyCrash
import com.instabug.early_crash.model.EarlyCrashToJsonMapper
import com.instabug.early_crash.model.EarlyCrashToMetadataMapper
import com.instabug.early_crash.network.ASynchronousSingleEarlyCrashUploader
import com.instabug.early_crash.network.AppStartupEarlyCrashUploaderJob
import com.instabug.early_crash.network.EarlyCrashRequestFactory
import com.instabug.early_crash.network.IEarlyCrashUploaderJob
import com.instabug.early_crash.network.NormalEarlyCrashUploaderJob
import com.instabug.early_crash.network.SingleEarlyCrashUploader
import com.instabug.early_crash.network.SynchronousSingleEarlyCrashUploader
import com.instabug.library.IBGNetworkWorker
import com.instabug.library.Instabug
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider
import com.instabug.library.internal.storage.AttachmentManager
import com.instabug.library.map.Mapper
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.networkv2.request.Request
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONObject
import java.util.concurrent.ExecutorService
import java.util.concurrent.Future

object EarlyCrashesServiceLocator {

    fun setEarlyContext(context: Context) {
        val applicationContext = context.applicationContext
        if (applicationContext is Application) {
            InstabugApplicationProvider.init(applicationContext)
        }
    }

    val earlyCrashToJsonMapper: Mapper<EarlyCrash, JSONObject>
        get() = EarlyCrashToJsonMapper()

    private val requestFactory: ParameterizedFactory<Request?, JSONObject>
        get() = EarlyCrashRequestFactory(
            SettingsManager.getInstance()
        )

    private val networkExecutorService: ExecutorService
        get() = PoolProvider.getNetworkingSingleThreadExecutorService(IBGNetworkWorker.CRASH)

    val cacheHandler: IEarlyCrashCacheHandler by lazy {
        EarlyCrashCacheHandler(
            directoryFactory = ReportsDirectory.Factory(
                Instabug::getApplicationContext,
                AttachmentManager::getAttachmentInternalDirectory
            )
        )
    }

    val configurationsProvider: IEarlyCrashesConfigProvider by lazy {
        EarlyCrashesConfigProvider(
            persistence = EarlyCrashesConfigPersistence(
                context = Instabug.getApplicationContext()
            ),
            CrashesServiceLocator.crashConfigurationProvider
        )
    }

    val configurationsHandler: ConfigurationsHandler by lazy {
        EarlyCrashesConfigHandler(
            configProvider = configurationsProvider
        )
    }

    private val singleCrashUploader: SingleEarlyCrashUploader<Runnable?>
        get() =
            SynchronousSingleEarlyCrashUploader(
                cacheHandler,
                requestFactory,
                NetworkManager(),
                CrashSettings.getInstance()
            )

    val asynchronousSingleCrashUploader: SingleEarlyCrashUploader<Future<Runnable?>>
        get() = ASynchronousSingleEarlyCrashUploader(
            singleCrashUploader,
            networkExecutorService
        )

    val normalEarlyCrashUploaderJob: IEarlyCrashUploaderJob
        get() = NormalEarlyCrashUploaderJob(
            cacheHandler,
            singleCrashUploader,
            networkExecutorService,
            EarlyCrashToMetadataMapper(),
            CommonsLocator.crashMetadataCallback
        )

    val startupEarlyCrashUploaderJob: IEarlyCrashUploaderJob
        get() = AppStartupEarlyCrashUploaderJob(
            cacheHandler,
            singleCrashUploader,
            networkExecutorService,
            3L
        )
}