package com.instabug.crash.cache

import com.instabug.commons.di.CommonsLocator
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.*
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper
import com.instabug.library.safetyUse
import java.util.regex.Matcher
import kotlin.math.min

private const val QUERY_ALIAS_PARTIAL_MESSAGE = "partial_message"
private const val QUERY_ALIAS_MESSAGE_LENGTH = "message_length"

private const val COLUMN_MESSAGE_LENGTH =
    "length($COLUMN_CRASH_MESSAGE) as $QUERY_ALIAS_MESSAGE_LENGTH"
private const val COLUMN_PARTIAL_MESSAGE =
    "substr($COLUMN_CRASH_MESSAGE, %d, %d) as $QUERY_ALIAS_PARTIAL_MESSAGE"

private const val REGEX_DROP_COUNTS = "(\"droppedThreads\":\\d+),(\"terminatedThreads\":\\d+)"

private const val LENGTH_THRESHOLD = 150_000L
private const val FORWARD_RETRIEVAL_LIMIT = 50_000L
private const val BACKWARD_RETRIEVAL_LIMIT = 10_000L

/**
 * An instance of this class should be used to retrieve crash messages.
 * If the message exceeds the [LENGTH_THRESHOLD], it will be trimmed to 200 frames limit.
 */
class CachedCrashMessageLimiter {
    private val messageBuilder = StringBuilder()
    private var processingStackTrace = false
    private var stackTraceCount = 0
    private var stackTraceFinalized = false

    /**
     * Retrieves crash message given its id.
     * If the crash message stored exceeds [LENGTH_THRESHOLD], a trimmed version of the message
     * will be returned, else the full message will be returned.
     * Trimming is done by limiting stackTrace to [ERROR_THREAD_FRAMES_LIMIT].
     * @param id A [String] representing the id of the crash.
     * @param database The open [SQLiteDatabaseWrapper] to be used to interact with an instance of Instabug's DB.
     * @return The full crash message if its length doesn't exceed [LENGTH_THRESHOLD] or else a limited version of it.
     * @throws Exception if any error happened while retrieving message.
     */
    @Throws(Exception::class)
    fun getCrashMessage(id: String, database: SQLiteDatabaseWrapper): String {
        val messageLength = getMessageLength(id, database)
        if (messageLength <= LENGTH_THRESHOLD) {
            return getFullMessage(id, database)
        }
        return getLimitedMessage(id, database, messageLength)
    }

    private fun getMessageLength(id: String, database: SQLiteDatabaseWrapper): Long =
        database.kQueryCrashesById(id, arrayOf(COLUMN_MESSAGE_LENGTH))?.safetyUse{ cursor ->
                require(cursor.moveToFirst())
                with(cursor) { getLong(getColumnIndexOrThrow(QUERY_ALIAS_MESSAGE_LENGTH)) }

        } ?: error("Cursor is null while retrieving message length")

    private fun getFullMessage(id: String, database: SQLiteDatabaseWrapper): String =
        database.kQueryCrashesById(id, arrayOf(COLUMN_CRASH_MESSAGE))?.safetyUse{ cursor ->

                require(cursor.moveToFirst())
                with(cursor) { getString(getColumnIndexOrThrow(COLUMN_CRASH_MESSAGE)) }

        } ?: error("Something went wrong while retrieving crash $id message.")

    private fun getLimitedMessage(
        id: String,
        database: SQLiteDatabaseWrapper,
        messageLength: Long
    ): String {
        processMessageForwardToStackTraceEnd(id, database, messageLength)
        processMessageBackwardForDropCounts(id, database, messageLength)
        val crashMessage = messageBuilder.toString()
        messageBuilder.clear()
        return crashMessage
    }

    /**
     * Process the crash message from the beginning with a step each retrieving [FORWARD_RETRIEVAL_LIMIT] char.
     * The forward processing stops when retrieving [ERROR_THREAD_FRAMES_LIMIT] frames of the stackTrace
     * or when the stackTrace attribute finishes.
     * @param id The id of the crash which message will be processed
     * @param database A [SQLiteDatabaseWrapper] instance to be used for interacting with the Instabug's database.
     * @param messageLength The original message length in database.
     */
    private fun processMessageForwardToStackTraceEnd(
        id: String,
        database: SQLiteDatabaseWrapper,
        messageLength: Long
    ) {
        var retrievedAmount = 0L
        while (!stackTraceFinalized && retrievedAmount < messageLength) {
            val delta = min(messageLength - retrievedAmount, FORWARD_RETRIEVAL_LIMIT)
            database.retrievePartialMessage(
                id,
                startOffset = retrievedAmount + 1,
                endOffset = retrievedAmount + delta
            ).also(this::appendStackTrace)
            retrievedAmount += delta
        }
        messageBuilder.append("\"},")
    }

    private fun appendStackTrace(patchString: String) {
        val indexOfFirstCharAfterStackTraceKey = patchString.indexOf("\"stackTrace\":\"") + 14

        val startIndex = if (!processingStackTrace) indexOfFirstCharAfterStackTraceKey else 0

        for ((index, char) in patchString.withIndex()) {
            val isTabAhead = char == '\\' && patchString[index + 1] == 't'
            if (index < startIndex || (!isTabAhead && char != '"')) {
                messageBuilder.append(char)
                continue
            }
            stackTraceCount++
            if (char == '"' || stackTraceCount > CommonsLocator.threadingLimitsProvider.provideErrorThreadFramesLimit()) {
                stackTraceFinalized = true
                processingStackTrace = false
                break
            }
            processingStackTrace = true
            messageBuilder.append(char)
        }
    }

    /**
     * Processes the crash message backwards to retrieved drop counts (droppedThreads & terminatedThreads).
     * The processing is limited to retrieving [BACKWARD_RETRIEVAL_LIMIT] char and search for the mentioned attributes.
     * If the drop counts are not found, the message is finalized.
     * @param id The id of the crash which message is being processed.
     * @param database An instance of [SQLiteDatabaseWrapper] to be used while interacting with Instabug's database.
     * @param messageLength The original message length in database.
     */
    private fun processMessageBackwardForDropCounts(
        id: String,
        database: SQLiteDatabaseWrapper,
        messageLength: Long
    ) {
        runCatching {
            database.retrievePartialMessage(
                id,
                startOffset = messageLength - BACKWARD_RETRIEVAL_LIMIT,
                endOffset = messageLength
            ).also(this::appendDropCounts)
        }.onFailure { messageBuilder.append("}") }
    }

    private fun appendDropCounts(patchString: String) {
        REGEX_DROP_COUNTS.toPattern().matcher(patchString)
            .apply(Matcher::find).run {
                group(1)?.let { messageBuilder.append("$it,") }
                group(2)?.let { messageBuilder.append("$it}") }
            }
    }

    private fun SQLiteDatabaseWrapper.retrievePartialMessage(
        id: String,
        startOffset: Long,
        endOffset: Long
    ): String {
        val columns = arrayOf(COLUMN_PARTIAL_MESSAGE.format(startOffset, endOffset))
        return kQueryCrashesById(id, columns)?.safetyUse { cursor ->
            require(cursor.moveToFirst())
            with(cursor) { getString(getColumnIndexOrThrow(QUERY_ALIAS_PARTIAL_MESSAGE)) }
        } ?: error("Something went wrong retrieving partial message for crash $id")
    }

    private fun SQLiteDatabaseWrapper.kQueryCrashesById(
        id: String,
        columns: Array<out String>
    ) = query(TABLE_NAME, columns, "$COLUMN_ID = ?", arrayOf(id), null, null, null)
}