package com.instabug.early_crash.model

import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.utils.hasRateLimitedPrefix
import com.instabug.crash.models.CrashMetadata
import com.instabug.library.map.Mapper
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.util.extenstions.toMap
import org.json.JSONObject

class EarlyCrashToMetadataMapper : Mapper<Pair<JSONObject, RequestResponse?>, CrashMetadata?> {

    override fun map(from: Pair<JSONObject, RequestResponse?>): CrashMetadata? =
        runCatching {
            from.second?.getTemporaryToken()?.let { token ->
                val errorJsonObject = from.first.getError()
                CrashMetadata(
                    occurrenceId = token,
                    errorType =  CRASH_TYPE,
                    errorDescription = errorJsonObject?.getDescription(),
                    errorCode = errorJsonObject?.getName().orEmpty(),
                    userAttributes = from.first.getUserAttributes(),
                    rateLimited = hasRateLimitedPrefix(token)
                )
            }
        }.getOrReportError(
            null,
            "Failed to map EarlyCrash to CrashMetadata",
            true
        )

    private fun RequestResponse.getTemporaryToken() =
        responseBody?.takeIf { it is String }
            ?.let { JSONObject(it as String) }
            ?.getNullableString("id")

    private fun JSONObject.getError(): JSONObject? =
        getNullableString(TITLE_KEY)
            ?.let { JSONObject(it) }
            ?.optJSONObject(ERROR_KEY)

    private fun JSONObject.getDescription(): String? =
        getNullableString(MESSAGE_KEY)

    private fun JSONObject.getName(): String? =
        getNullableString(NAME_KEY)

    private fun JSONObject.getUserAttributes(): Map<String, String>? =
        optJSONObject(USER_ATTRIBUTES_KEY)?.toMap()?.mapValues { it.value.toString() }

    private fun JSONObject.getNullableString(key: String): String? = runCatching {
        getString(key)
    }.getOrNull()

    companion object {
        private const val CRASH_TYPE = "Crash"
        private const val TITLE_KEY = "title"
        private const val ERROR_KEY = "error"
        private const val MESSAGE_KEY = "message"
        private const val NAME_KEY = "name"
        private const val USER_ATTRIBUTES_KEY = "user_attributes"
    }
}