package com.instabug.crash.utils;

import static com.instabug.library.Instabug.getApplicationContext;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.WorkerThread;

import com.instabug.anr.cache.AnrReportsDbHelper;
import com.instabug.crash.Constants;
import com.instabug.crash.CrashReporting;
import com.instabug.crash.cache.CrashReportsDbHelper;
import com.instabug.crash.models.Crash;
import com.instabug.library.Instabug;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation;
import com.instabug.library.model.Attachment;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.util.List;

public class CrashCleanupUtils {

    private static final String TAG = "CrashCleanupUtils";
    private static final String CRASH_STATE_FILES_PREFIX = "files:crash_state:";
    private static final String ANR_STATE_FILES_PREFIX = "files:anr_state:";

    @WorkerThread
    public static synchronized void cleanStaleStateFiles() {
        List<File> crashStateFiles = FileUtils.getStateFiles(CRASH_STATE_FILES_PREFIX);
        if (!crashStateFiles.isEmpty()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Found " + crashStateFiles.size() + " stale crash state files on disk, cleaning ...");
            List<String> cachedCrashesStateFiles = CrashReportsDbHelper.getCrashesStateFiles();
            for (File stateFile : crashStateFiles) {
                try {
                    boolean isFileExistsInDB = false;
                    for (String filename : cachedCrashesStateFiles) {
                        if (filename.contains(stateFile
                                .getName()
                                .substring(stateFile.getName()
                                        .indexOf(CrashReporting.CRASH_STATE)
                                        + CrashReporting.CRASH_STATE.length() + 1))) {
                            isFileExistsInDB = true;
                            break;
                        }
                    }
                    if (!isFileExistsInDB) {
                        boolean isDeleted = stateFile.delete();
                        if (isDeleted) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file " + stateFile.getName() + " is deleted");
                        } else {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file " + stateFile.getName() + " is not deleted");
                        }
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while cleaning stale ANR state files");
                    IBGDiagnostics.reportNonFatal(e, "can't clean Stale Crashes State Files");
                }
            }
        }
    }

    public static synchronized void cleanStaleANRStateFiles() {
        List<File> crashStateFiles = FileUtils.getStateFiles(ANR_STATE_FILES_PREFIX);
        if (!crashStateFiles.isEmpty()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Found " + crashStateFiles.size() + " stale ANR state files on disk, cleaning ...");
            List<String> cachedCrashesStateFiles = AnrReportsDbHelper.getANRsStateFiles();
            for (File stateFile : crashStateFiles) {
                try {
                    boolean isFileExistsInDB = false;
                    for (String filename : cachedCrashesStateFiles) {
                        if (filename != null && filename.contains(stateFile
                                .getName()
                                .substring(stateFile.getName()
                                        .indexOf(CrashReporting.ANR_STATE)
                                        + CrashReporting.ANR_STATE.length() + 1))) {
                            isFileExistsInDB = true;
                            break;
                        }
                    }
                    if (!isFileExistsInDB) {
                        boolean isDeleted = stateFile.delete();
                        if (isDeleted) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file " + stateFile.getName() + " is deleted");
                        } else {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file " + stateFile.getName() + " is not deleted");
                        }
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while cleaning stale ANR state files");
                    IBGDiagnostics.reportNonFatal(e, "can't clean Stale ANR State Files");
                }
            }
        }
    }

    public static synchronized void deleteAllCrashesAsync() {
        PoolProvider.postIOTask(() -> {
            Context context = getApplicationContext();
            if (context != null) {
                List<String> crashesIdsList = CrashReportsDbHelper.retrieveIds();
                // Remove the first crash occurrence from the list till it reaches MAX_CRASHES_ALLOWED
                while (crashesIdsList.size() > 0) {
                    String targetCrashId = crashesIdsList.get(0);
                    Crash crash = CrashReportsDbHelper.retrieveById(targetCrashId, getApplicationContext());
                    if (crash == null) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving crash " + targetCrashId + " while deleting all crashes");
                        crashesIdsList.remove(0);
                        continue;
                    }
                    if (crash.getState() != null && crash.getState().getUri() != null) {
                        deleteCrashStateUriFile(crash.getState().getUri());
                    }
                    deleteCrashAttachments(crash);
                    if (crash.getId() != null) {
                        CrashReportsDbHelper.delete(crash.getId());
                    }
                    crashesIdsList.remove(0);
                }
            }
        });
    }


    @WorkerThread
    public synchronized static void deleteCrashStateUriFile(Uri uri) {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            try {
                DiskUtils.with(context)
                        .deleteOperation(new DeleteUriDiskOperation(uri)).execute();
            } catch (Exception | OutOfMemoryError e) {
                IBGDiagnostics.reportNonFatal(e, "crash state file couldn't be removed");
            }
        }
    }

    @WorkerThread
    public synchronized static void deleteCrashAttachments(Crash crash) {
        List<Attachment> attachments = crash.getAttachments();
        for (Attachment attachment : attachments) {
            if (attachment.getLocalPath() != null && attachment.getName() != null) {
                // Delete the attachment file
                File attachmentFile = new File(attachment.getLocalPath());
                attachmentFile.delete();
                // Delete the attachment instance from DB
                if (attachment.getId() != -1) {
                    AttachmentsDbHelper.delete(attachment.getId());
                } else {
                    if (crash.getId() != null) {
                        AttachmentsDbHelper.delete(attachment.getName(), crash.getId());
                    } else {
                        InstabugSDKLogger.e(TAG, "Couldn't delete attachments: crash.getId() is null");
                    }
                }
            }
        }
    }
}
