package com.instabug.commons.session

import android.database.Cursor
import com.instabug.commons.models.Incident.Type
import com.instabug.commons.session.SessionIncident.ValidationStatus
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.storage.cache.dbv2.*
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionIncidentEntry
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.terminations.Constants

interface SessionIncidentCachingHandler {
    fun store(sessionIncident: SessionIncident)
    fun updateValidationStatus(incidentId: String, @ValidationStatus status: Int)
    fun updateValidationStatus(
        sessionId: String,
        incidentId: String?,
        incidentType: Type,
        @ValidationStatus status: Int
    )
    fun cleanseByIncidentType(incidentType: Type)
    fun trim(sessionId: String, incidentType: Type, limit: Int)
    fun delete(sessionIncident: SessionIncident)
    fun queryIncidentsBySessionsIds(sessionsIds: List<String>): List<SessionIncident>
    fun deleteIncidentsBySessionsIds(sessionsIds: List<String>)
}

class DefaultSessionIncidentCachingHandler : SessionIncidentCachingHandler {

    private val databaseManager: IBGDbManager
        get() = IBGDbManager.getInstance()

    override fun store(sessionIncident: SessionIncident) = runCatching {
        databaseManager.insert(
            SessionIncidentEntry.TABLE_NAME,
            null,
            sessionIncident.toContentValues()
        ); Unit
    }.getOrReport(Unit, "Failed to store session incident")

    override fun updateValidationStatus(incidentId: String, status: Int) = runCatching {
        databaseManager.update(
            SessionIncidentEntry.TABLE_NAME,
            IBGContentValues().apply {
                put(SessionIncidentEntry.COLUMN_VALIDATION_STATUS, status, true)
            },
            "${SessionIncidentEntry.COLUMN_INCIDENT_ID} = ?",
            listOf(IBGWhereArg(incidentId, true))
        ); Unit
    }.getOrReport(Unit, "Failed to validate Session-Incident link by incident Id: $incidentId")

    override fun updateValidationStatus(
        sessionId: String,
        incidentId: String?,
        incidentType: Type,
        status: Int
    ) = runCatching {
        databaseManager.update(
            SessionIncidentEntry.TABLE_NAME,
            IBGContentValues().apply {
                put(SessionIncidentEntry.COLUMN_INCIDENT_ID, incidentId, true)
                put(SessionIncidentEntry.COLUMN_VALIDATION_STATUS, status, true)
            },
            "${SessionIncidentEntry.COLUMN_SESSION_ID} = ? AND ${SessionIncidentEntry.COLUMN_INCIDENT_TYPE} = ?",
            listOf(IBGWhereArg(sessionId, true), IBGWhereArg(incidentType.name, true))
        ); Unit
    }.getOrReport(Unit, "Failed to validate Session-Incident link by incident Id: $incidentId")

    override fun cleanseByIncidentType(incidentType: Type) = runCatching {
        databaseManager.kDelete(
            SessionIncidentEntry.TABLE_NAME,
            "${SessionIncidentEntry.COLUMN_INCIDENT_TYPE} = ? AND" +
                " ${SessionIncidentEntry.COLUMN_VALIDATION_STATUS} = ?",
            listOf(
                IBGWhereArg(incidentType.name, true),
                IBGWhereArg(ValidationStatus.UNVALIDATED.toString(), true)
            )
        ); Unit
    }.getOrReport(Unit, "Failed to cleanse Session-Incident links by type: $incidentType")

    override fun trim(sessionId: String, incidentType: Type, limit: Int) = runCatching {
        databaseManager.kDelete(
            table = SessionIncidentEntry.TABLE_NAME,
            selection = SessionIncidentEntry.TRIM_WHERE_CLAUSE,
            selectionArgs = listOf(
                IBGWhereArg(sessionId, true),
                IBGWhereArg(incidentType.name, true),
                IBGWhereArg("-1", true),
                IBGWhereArg(limit.toString(), true)
            )
        ); Unit
    }.getOrReport(Unit, "Failed to trim session incidents")

    override fun delete(sessionIncident: SessionIncident) = runCatching {
        databaseManager.kDelete(
            SessionIncidentEntry.TABLE_NAME, "${SessionIncidentEntry.COLUMN_ID} = ?",
            listOf(IBGWhereArg(sessionIncident.id.toString(), true))
        ); Unit
    }.getOrReport(Unit, "Failed to delete session incident with id: ${sessionIncident.id}")

    override fun queryIncidentsBySessionsIds(sessionsIds: List<String>): List<SessionIncident> =
        runCatching {
            databaseManager.kQuery(
                table = SessionIncidentEntry.TABLE_NAME,
                whereClause = whereSessionIdIn(sessionsIds)
            )?.toIncidentList().orEmpty()
        }.getOrReport(listOf(), "Failed to query incidents by sessions ids")

    private fun whereSessionIdIn(sessionsIds: List<String>): Pair<String, List<IBGWhereArg>> {
        val whereStmt =
            "${SessionIncidentEntry.COLUMN_SESSION_ID} IN ${sessionsIds.joinToArgs()}"
        val whereArgs = sessionsIds.asArgs()
        return Pair(whereStmt, whereArgs)
    }

    override fun deleteIncidentsBySessionsIds(sessionsIds: List<String>) {
        kotlin.runCatching {
            val whereClause = whereSessionIdIn(sessionsIds)
            databaseManager.kDelete(
                SessionIncidentEntry.TABLE_NAME, selection = whereClause.selection,
                selectionArgs = whereClause.args
            )
        }.getOrReport(Unit, "Failed to delete incidents by sessions ids ")
    }

    private fun IBGCursor.toIncidentList() = use {
        buildList {
            while (moveToNext()) add(toSessionIncident())
        }
    }

    private fun SessionIncident.toContentValues() = IBGContentValues().apply {
        put(SessionIncidentEntry.COLUMN_ID, id, true)
        put(SessionIncidentEntry.COLUMN_SESSION_ID, sessionId, true)
        put(SessionIncidentEntry.COLUMN_INCIDENT_ID, incidentId, true)
        put(SessionIncidentEntry.COLUMN_INCIDENT_TYPE, incidentType.name, true)
        put(SessionIncidentEntry.COLUMN_VALIDATION_STATUS, validationStatus, true)
    }

    private fun Cursor.toSessionIncident() = SessionIncident(
        id = getLong(getColumnIndexOrThrow(SessionIncidentEntry.COLUMN_ID)),
        sessionId = getString(getColumnIndexOrThrow(SessionIncidentEntry.COLUMN_SESSION_ID)),
        incidentId = getString(getColumnIndexOrThrow(SessionIncidentEntry.COLUMN_INCIDENT_ID)),
        incidentType = Type.valueOf(getString(getColumnIndexOrThrow(SessionIncidentEntry.COLUMN_INCIDENT_TYPE))),
        validationStatus = getInt(getColumnIndexOrThrow(SessionIncidentEntry.COLUMN_VALIDATION_STATUS))
    )

    private fun <R> Result<R>.getOrReport(default: R, message: String) = getOrElse { throwable ->
        InstabugSDKLogger.e(Constants.LOG_TAG, message, throwable)
        InstabugCore.reportError(throwable, message)
        default
    }
}
