package com.instabug.bganr

import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.anr.di.AnrServiceLocator
import com.instabug.anr.di.AnrServiceLocator.anrConfigurationHandler
import com.instabug.anr.di.AnrServiceLocator.anrConfigurationProvider
import com.instabug.anr.model.Anr
import com.instabug.bganr.BackgroundAnrLocator.appCtx
import com.instabug.bganr.BackgroundAnrLocator.backgroundAnrCacheDir
import com.instabug.bganr.BackgroundAnrLocator.captorsRegistry
import com.instabug.bganr.BackgroundAnrLocator.configurationsHandler
import com.instabug.bganr.BackgroundAnrLocator.configurationsProvider
import com.instabug.bganr.BackgroundAnrLocator.crashesCacheDir
import com.instabug.bganr.BackgroundAnrLocator.hubDataWatcher
import com.instabug.bganr.BackgroundAnrLocator.reproScreenshotsCacheDir
import com.instabug.bganr.BackgroundAnrLocator.sessionLinker
import com.instabug.bganr.BackgroundAnrLocator.syncJob
import com.instabug.commons.PluginDelegate
import com.instabug.commons.di.CommonsLocator.reproProxy
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.logging.logWarning
import com.instabug.commons.logging.runOrReportError
import com.instabug.commons.models.Incident.Type
import com.instabug.commons.snapshot.StateSnapshotCaptor
import com.instabug.commons.utils.isAtLeastRunningR
import com.instabug.crash.Constants.Preferences
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.sessionV3.sync.AllFilter
import com.instabug.library.sessionV3.sync.NoneFilter
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider
import java.io.File

@VisibleForTesting
const val SNAPSHOT_SYSTEM_ID = 0x003

@VisibleForTesting
const val ERROR_BG_ANR_NOT_SUPPORTED =
    "Instabug Background ANR is disabled because It's supported starting from Android 11."

@VisibleForTesting
const val ERROR_BG_ANR_DISABLED_BE =
    "Background ANR wasn't enabled as the feature seems to be disabled for your Instabug company account. " +
            "Please contact support for more information."

@VisibleForTesting
const val OP_EXEC_QUEUE_ID = "bg-anr-op"

class BackgroundAnrPluginDelegate : PluginDelegate {

    private var isLastEnabled: Boolean =
        DEFAULT_BG_ANR_AVAILABILITY || Preferences.ANR_V2_AVAILABLE.second

    override fun init(context: Context) {
        if (!isAtLeastRunningR) {
            ERROR_BG_ANR_NOT_SUPPORTED.logWarning()
            return
        }
        addMutualDirsWatcher()
    }

    override fun start(context: Context) {
        if (!isAtLeastRunningR) return
        operateOnExec {
            isLastEnabled = isAnrV2OrBgAnrEnabled()
            "ANRs-V2 -> Initial state = $isLastEnabled".logVerbose()
            if (!isLastEnabled) removeMutualDirsWatcher()
            SettingsManager.getInstance()
                ?.reproConfigurations
                ?.let { handleReproStateConfigurations(it.modesMap) }
            if (!configurationsProvider.isAvailable) ERROR_BG_ANR_DISABLED_BE.logWarning()
            if (configurationsProvider.isAnrV2Enabled)
                appCtx?.let(this::migrateAndSync)

        }
    }

    private fun isAnrV2OrBgAnrEnabled() =
        configurationsProvider.isEnabled || configurationsProvider.isAnrV2Enabled

    override fun wake() {
        if (!isAtLeastRunningR) return
        "ANRs-V2 -> Plugin is waking..".logVerbose()
        operateOnExec {
            if (isLastEnabled) {
                createSessionWeakLink()
                startSnapshotCaptorIfPossible()
                createCurrentSessionBaseLine()
                appCtx?.let(this::migrateAndSync)

            }
        }
    }

    private fun createCurrentSessionBaseLine() {
        crashesCacheDir.currentSessionDirectory
            .also { "ANRs-V2 -> Current session id: ${it?.name}".logVerbose() }
            ?.run(this::markCurrentSessionWithBaseline)
    }

    private fun markCurrentSessionWithBaseline(currentSessionDir: File) {
        runCatching {
            "ANRs-V2 -> Creating baseline file for session ${currentSessionDir.name}".logVerbose()
            BackgroundAnrCacheDir.createTraceBaselineFile(currentSessionDir)
        }.runOrReportError("ANRs-V2 -> Couldn't create baseline file for current session.")
    }

    override fun sleep() {
        // No-Op
    }

    override fun stop() {
        if (!isAtLeastRunningR) return
        operateOnExec {
            stopSnapshotCaptor()
            consentOnMutualDirsCleansing()
        }
    }

    override fun handleSDKCoreEvent(sdkCoreEvent: IBGSdkCoreEvent) {
        if (!isAtLeastRunningR) return
        when (sdkCoreEvent) {
            is IBGSdkCoreEvent.FeaturesFetched -> {
                "ANRs-V2 -> received features fetched".logVerbose()
                operateOnExec {
                    configurationsHandler.handleConfiguration(sdkCoreEvent.response)
                    onFeatureStateChange()
                }
            }

            is IBGSdkCoreEvent.Features -> {
                "ANRs-V2 -> received features".logVerbose()
                operateOnExec(this::onFeatureStateChange)
            }

            is IBGSdkCoreEvent.ReproState -> handleReproStateConfigurations(sdkCoreEvent.modesMap)

            is IBGSdkCoreEvent.NetworkActivated -> {
                "ANRs-V2 -> received network activated".logVerbose()
                operateOnExec { startSyncingIfPossible() }
            }

            else -> Unit
        }
    }

    private fun operateOnExec(operation: () -> Unit) {
        PoolProvider.postOrderedIOTask(OP_EXEC_QUEUE_ID, operation)
    }

    @WorkerThread
    private fun startSnapshotCaptorIfPossible() {
        captorsRegistry.start(SNAPSHOT_SYSTEM_ID, StateSnapshotCaptor.Factory())
    }

    @WorkerThread
    private fun stopSnapshotCaptor() {
        captorsRegistry.stop(SNAPSHOT_SYSTEM_ID, StateSnapshotCaptor.ID)
    }

    @RequiresApi(Build.VERSION_CODES.R)
    @WorkerThread
    private fun onFeatureStateChange() {
        if (isAnrV2OrBgAnrEnabled() == isLastEnabled) return
        if (isAnrV2OrBgAnrEnabled()) {
            isLastEnabled = true
            "ANRs-V2 -> enabled".logVerbose()
            createSessionWeakLink()
            startSnapshotCaptorIfPossible()
            createCurrentSessionBaseLine()

            appCtx?.let(this::migrateAndSync)
            addMutualDirsWatcher()
            return
        }
        isLastEnabled = false
        "ANRs-V2 -> disabled".logVerbose()
        validateCurrentSessionWeakLink()
        stopSnapshotCaptor()
        backgroundAnrCacheDir.deleteFileDir()
        removeMutualDirsWatcher()
        if (!configurationsProvider.isAvailable) ERROR_BG_ANR_DISABLED_BE.logWarning()
    }

    @RequiresApi(Build.VERSION_CODES.R)
    @WorkerThread
    private fun migrateAndSync(ctx: Context): MigrationResult = with(BackgroundAnrLocator) {
        migrator(ctx)
            .also { "ANRs-V2 -> migration result $it".logVerbose() }
            .also { consentOnMutualDirsCleansing() }
            .also(::validateWeakLinks)
            .also(::notifyDataReadiness)
            .also { AnrServiceLocator.earlyAnrMigrator(ctx, it.migratedTimeStamps) }
            .also { startSyncingIfPossible() }
    }

    @WorkerThread
    private fun createSessionWeakLink() {
        val runningSession = InstabugCore.getRunningSession()
        runningSession
            ?.takeIf { configurationsProvider.isEnabled }
            ?.also { session -> sessionLinker.weakLink(session.id, Type.BG_ANR) }
        runningSession
            ?.takeIf { configurationsProvider.isAnrV2Enabled }
            ?.also { session -> sessionLinker.weakLink(session.id, Type.ANR) }
    }

    @WorkerThread
    private fun validateCurrentSessionWeakLink() {
        val runningSession = InstabugCore.getRunningSession()
        runningSession
            ?.also { session -> sessionLinker.validateWeakLink(session.id, null, Type.BG_ANR) }
        runningSession
            ?.also { session -> sessionLinker.validateWeakLink(session.id, null, Type.ANR) }
    }

    @WorkerThread
    private fun validateWeakLinks(result: MigrationResult) {
        result.incidents.forEach { anr ->
            sessionLinker.validateWeakLink(anr.sessionId, anr.metadata.uuid, anr.type)
        }
        val sessionsWithIncident = result.incidents.map { it.sessionId }.toSet()
        (result.migratedSessions - sessionsWithIncident).forEach { sessionId ->
            sessionLinker.validateWeakLink(sessionId, null, Type.BG_ANR)
            sessionLinker.validateWeakLink(sessionId, null, Type.ANR)
        }
    }

    private fun validateWeakLink(anr: Anr) {
        sessionLinker.validateWeakLink(anr.sessionId, anr.metadata.uuid, anr.type)
        val isAnrV2Enabled = configurationsProvider.isAnrV2Enabled
        val isBGAnrEnabled = configurationsProvider.isEnabled
        if ((anr.type == Type.BG_ANR && isAnrV2Enabled) || (anr.type == Type.ANR && isBGAnrEnabled)) {
            val targetType = if (anr.type == Type.BG_ANR) Type.ANR else Type.BG_ANR
            sessionLinker.validateWeakLink(anr.sessionId, null, targetType)
        }
    }

    @WorkerThread
    private fun notifyDataReadiness(result: MigrationResult) {
        val batchingFilter =
            result.incidents.size.takeIf { it > 0 }?.let { NoneFilter } ?: AllFilter
        InstabugCore.notifyV3SessionDataReadiness(batchingFilter)
    }

    private fun startSyncingIfPossible() {
        if (!isLastEnabled) return
        syncJob.start()
    }

    private fun addMutualDirsWatcher() {
        crashesCacheDir.addWatcher(SNAPSHOT_SYSTEM_ID)
        reproScreenshotsCacheDir.addWatcher(SNAPSHOT_SYSTEM_ID)
        hubDataWatcher.addWatcher(SNAPSHOT_SYSTEM_ID)
    }

    private fun removeMutualDirsWatcher() {
        crashesCacheDir.removeWatcher(SNAPSHOT_SYSTEM_ID)
        reproScreenshotsCacheDir.removeWatcher(SNAPSHOT_SYSTEM_ID)
        hubDataWatcher.removeWatcher(SNAPSHOT_SYSTEM_ID)
    }

    private fun consentOnMutualDirsCleansing() {
        crashesCacheDir.consentOnCleansing(SNAPSHOT_SYSTEM_ID)
        reproScreenshotsCacheDir.consentOnCleansing(SNAPSHOT_SYSTEM_ID)
        hubDataWatcher.consentOnCleansing(SNAPSHOT_SYSTEM_ID)
    }

    private fun handleReproStateConfigurations(modesMap: Map<Int, Int>) {
        anrConfigurationHandler.handle(modesMap)
        reproProxy.evaluate(anrConfigurationProvider)
    }
}
