package com.instabug.fatalhangs.configuration

import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.preferences.crashesPreferences
import com.instabug.commons.utils.SharedPreferencesUtils
import com.instabug.crash.Constants
import com.instabug.fatalhangs.di.FatalHangsServiceLocator
import com.instabug.library.Instabug
import com.instabug.library.IssueType
import com.instabug.library.ReproMode
import com.instabug.library.internal.sharedpreferences.corePreferences
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject

class FatalHangsConfigurationHandlerImpl : ConfigurationsHandler {
    companion object {
        private const val CRASHES_RESPONSE_KEY = "crashes"
        private const val FATAL_HANGS_RESPONSE_KEY = "android_fatal_hangs"
        const val FATAL_HANGS_CORE_KEY = "FATAL_HANGSAVAIL"
        private const val FALLBACK_FATAL_HANG_SENSITIVITY = 1000L
    }

    private val configurationProvider: FatalHangsConfigurationProvider by lazy {
        FatalHangsServiceLocator.fatalHangsConfigurationProvider
    }

    override fun handleConfiguration(configuration: String?) {
        if (!configurationsParsed(configuration))
            with(configurationProvider) {
                isFatalHangsAvailable = Constants.Preferences.FATAL_HANGS_AVAILABILITY.second
            }
    }

    private fun configurationsParsed(configuration: String?): Boolean {
        runOrLogError(errorMessage = "Something went wrong while parsing Fatal hangs from features response ") {
            configuration?.let { config ->
                with(JSONObject(config)) {
                    crashes?.let {
                        it.fatalHangs?.let { fatalHangs ->
                            val isAvailable = fatalHangs.isAvailable
                            val sensitivity = fatalHangs.sensitivity
                            with(configurationProvider) {
                                isFatalHangsAvailable = isAvailable
                                fatalHangsSensitivity = if (sensitivity <= 0) FALLBACK_FATAL_HANG_SENSITIVITY else sensitivity
                            }
                            return true
                        }
                    }
                }
            }
        }
        return false
    }

    override fun handle(modesMap: Map<Int, Int>) {
        modesMap[IssueType.AppHang]?.let { appHangMode ->
            with(configurationProvider) {
                isReproStepsEnabledSDK = appHangMode > ReproMode.Disable
                isReproScreenShotsEnabledSDK = appHangMode > ReproMode.EnableWithNoScreenshots
            }
        }
    }

    private val JSONObject.crashes
        get() = this.optJSONObject(CRASHES_RESPONSE_KEY)

    private val JSONObject.fatalHangs
        get() = this.optJSONObject(FATAL_HANGS_RESPONSE_KEY)

    private val JSONObject.isAvailable
        get() = this.optBoolean("v3_enabled")

    private val JSONObject.sensitivity
        get() = this.optLong("sensitivity_ms", 1000)

    override fun migrateCurrentConfiguration() {
        takeUnless { isFatalHangsMigrated() }
            ?.let {
                with(configurationProvider) {
                    Instabug.getApplicationContext()?.let {
                        SharedPreferencesUtils.readBoolean(
                            FATAL_HANGS_CORE_KEY,
                            Constants.Preferences.FATAL_HANGS_AVAILABILITY.second,
                            corePreferences
                        ).also { migratedValue ->
                            isFatalHangsAvailable = migratedValue
                            setFatalHangsMigrated()
                        }
                        fatalHangsSensitivity = SettingsManager.getFatalHangsSensitivity()
                    }
                }
            }
    }
    private fun isFatalHangsMigrated(): Boolean {
        val (key, defaultValue) = Constants.Preferences.IS_FATAL_HANGS_MIGRATED
        return SharedPreferencesUtils.readBoolean(key, defaultValue, crashesPreferences)
    }

    private fun setFatalHangsMigrated() {
        SharedPreferencesUtils.writeBoolean(
            Constants.Preferences.IS_FATAL_HANGS_MIGRATED.first,
            true,
            crashesPreferences
        )
    }

}