package com.instabug.early_crash.threading

import com.instabug.library.util.threading.OrderedExecutorService
import com.instabug.library.util.threading.PoolProvider
import java.util.concurrent.Callable

interface ExecutionMode {
    operator fun invoke(operation: () -> Unit)
    operator fun <Out> invoke(operation: () -> Out): Out

    companion object {
        val Immediate: ExecutionMode by lazy {
            object : ExecutionMode {
                override fun invoke(operation: () -> Unit) = run { operation() }
                override fun <Out> invoke(operation: () -> Out): Out = operation()
            }
        }

        val OrderedCaching: ExecutionMode by lazy {
            OrderingQueueExecutionMode(
                executor = PoolProvider.getInstance().orderedExecutor,
                execQId = ExecQs.Caching
            )
        }
    }
}

class OrderingQueueExecutionMode(
    private val executor: OrderedExecutorService,
    private val execQId: String
) : ExecutionMode {
    override fun invoke(operation: () -> Unit) {
        executor.execute(execQId, operation)
    }

    override fun <Out> invoke(operation: () -> Out): Out =
        executor.submit(execQId, Callable(operation)).get()
}