package com.instabug.crash.utils

import android.content.Context
import com.instabug.anr.cache.AnrReportsDbHelper
import com.instabug.anr.model.Anr
import com.instabug.commons.models.Incident
import com.instabug.crash.Constants
import com.instabug.crash.cache.CrashReportsDbHelper
import com.instabug.crash.models.Crash
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation
import com.instabug.library.internal.storage.operation.DiskOperationCallback
import com.instabug.library.model.Attachment
import com.instabug.library.util.InstabugSDKLogger
import java.io.File

//region Crashes
fun deleteCrash(context: Context, crash: Crash) {
    runCatching {
        crash.attachments
            .forEach { deleteAttachment(it, crash.id) }
            .also { deleteCrashAndStateFile(context, crash) }
    }.onFailure {
        InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't delete crash ${crash.id}", it)
    }
}

fun deleteCrashAndStateFile(context: Context, crash: Crash) {
    crash.state?.uri
        ?.let { crash.deleteStateFile(context) }
        ?: crash.apply {
            InstabugSDKLogger.v(Constants.LOG_TAG, "No state file found. deleting the crash")
            deleteSavingDir(context)
            delete()
        }
}

private fun Crash.deleteStateFile(context: Context) {
    InstabugSDKLogger.v(
        Constants.LOG_TAG,
        "attempting to delete state file for crash with id: $id"
    )
    DiskUtils.with(context)
        .deleteOperation(DeleteUriDiskOperation(state!!.uri))
        .executeAsync(object : DiskOperationCallback<Boolean?> {
            override fun onSuccess(args: Boolean?) {
                //deleting crash from db
                InstabugSDKLogger.v(
                    Constants.LOG_TAG,
                    "deleting crash:$id"
                )
                deleteSavingDir(context)
                delete()
            }

            override fun onFailure(t: Throwable) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error " + t.message + " while deleting crash state file"
                )
            }
        })
}

private fun Crash.delete() {
    if (id != null) CrashReportsDbHelper.delete(id)
}

//endregion
//region ANR
fun deleteAnr(context: Context, anr: Anr) {
    runCatching {
        anr.attachments
            .forEach { deleteAttachment(it, anr.id) }
            .also { deleteAnrAndStateFile(context, anr) }
    }.onFailure {
        InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't delete anr ${anr.id}", it)
    }
}

fun deleteAnrAndStateFile(context: Context, anr: Anr) {
    anr.state?.uri
        ?.let { anr.deleteStateFile(context) }
        ?: anr.apply {
            InstabugSDKLogger.e(Constants.LOG_TAG, "No state file found. deleting ANR")
            deleteSavingDir(context)
            delete()
        }
}

fun Anr.deleteStateFile(context: Context) {
    InstabugSDKLogger.v(
        Constants.LOG_TAG,
        "attempting to delete state file for ANR with id: $id"
    )

    if (
        DiskUtils.with(context)
            .deleteOperation(DeleteUriDiskOperation(state.uri))
            .execute()
    ) {
        deleteSavingDir(context)
        delete()
    } else {
        InstabugSDKLogger.e(Constants.LOG_TAG, "Error while deleting ANR state file")
    }
}

fun Anr.delete() {
    AnrReportsDbHelper.delete(id)
}

//endregion

//region incident
fun Incident.deleteSavingDir(ctx: Context) {
    getSavingDirOnDisk(ctx).takeIf(File::exists)?.deleteRecursively()
}
//endregion

//region Attachments
fun deleteAttachment(attachment: Attachment, ownerId: String?) {
    attachment.localPath
        ?.let(::File)
        ?.delete()
        ?.also(attachment::logAttachmentRemovedOrNot)
        ?.also { attachment.deleteFromDatabase(ownerId) }
}

private fun Attachment.logAttachmentRemovedOrNot(deleted: Boolean) {
    if (!deleted) {
        InstabugSDKLogger.w(Constants.LOG_TAG, "Attachment: $this is not removed")
    } else {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Attachment: $this is removed")
    }
}

private fun Attachment.deleteFromDatabase(ownerId: String?) {
    if (id != -1L) {
        AttachmentsDbHelper.delete(id)
    } else {
        if (name != null && ownerId != null) {
            AttachmentsDbHelper.delete(name, ownerId)
        }
    }
}
//endregion