package com.instabug.anr.early_anr

import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.anr.cache.AnrReportsDbHelper
import com.instabug.anr.configuration.AnrConfigurationProvider
import com.instabug.anr.model.Anr
import com.instabug.commons.OSExitInfo
import com.instabug.commons.OSExitInfoExtractor
import com.instabug.commons.isAnr
import com.instabug.commons.isExplicitlyInForeground
import com.instabug.commons.logging.logDebug
import com.instabug.commons.logging.runOrReportError
import com.instabug.commons.models.IncidentMetadata
import com.instabug.commons.utils.updateScreenShotAnalytics
import com.instabug.crash.Constants.EARLY_ANR_EXIT_INFO_COUNT
import com.instabug.crash.Constants.Preferences.LAST_EARLY_ANR_MIGRATION_MILLS
import com.instabug.library.core.InstabugCore
import com.instabug.library.model.State
import com.instabug.library.util.TimeUtils

fun interface IEarlyAnrMigrator {
    operator fun invoke(ctx: Context, migratedAnrsTimestamps: List<Long>)
}

@RequiresApi(Build.VERSION_CODES.R)
class EarlyAnrMigrator(
    private val configurationsProvider: AnrConfigurationProvider,
    private val infoExtractor: OSExitInfoExtractor
) : IEarlyAnrMigrator {


    override fun invoke(ctx: Context, migratedAnrsTimestamps: List<Long>) {
        runCatching {
            infoExtractor
                .takeIf { configurationsProvider.isAnrV2Enabled }
                ?.takeIf { InstabugCore.isLastSDKStateEnabled(ctx) }
                ?.also { "ANRs-V2 -> getting exit info".logDebug() }
                ?.getInfoList(ctx)
                ?.also { "ANRs-V2 -> filtering exit info list $it".logDebug() }
                ?.filter { it.isAnr() && it.isExplicitlyInForeground() }
                ?.filterNot { migratedAnrsTimestamps.contains(it.timestamp) }
                ?.mapNotNull { createEarlyAnr(it, ctx) }
                ?.forEach(AnrReportsDbHelper::insert)
            updateLastMigrationTime()
        }.runOrReportError("ANRs-V2 -> something went wrong while capturing early anr", true)
    }

    private fun OSExitInfoExtractor.getInfoList(ctx: Context): List<OSExitInfo> =
        if (isLastMigrationTimeHasDefaultValue())
            extractByCount(ctx, EARLY_ANR_EXIT_INFO_COUNT)
        else
            extract(ctx, configurationsProvider.lastEarlyAnrMigrationMillis).infoList

    private fun createEarlyAnr(
        info: OSExitInfo, ctx: Context
    ): Anr? {
        val stream = info.traceStream() ?: return null
        "ANRs-V2 -> found Anr $info and added for sync ".logDebug()
        val state = State.getNonChangingState(ctx)
        state.updateScreenShotAnalytics()
        return Anr.Factory().createEarlyAnr(
            ctx,
            stream,
            state,
            IncidentMetadata.Factory.create(),
        )
    }

    private fun updateLastMigrationTime() {
        configurationsProvider.lastEarlyAnrMigrationMillis = TimeUtils.currentTimeMillis()
    }

    private fun isLastMigrationTimeHasDefaultValue() =
        (configurationsProvider.lastEarlyAnrMigrationMillis == LAST_EARLY_ANR_MIGRATION_MILLS.second).also { hasDefaultValue ->
            if (hasDefaultValue) "ANRs-V2 -> last migration time is ${configurationsProvider.lastEarlyAnrMigrationMillis}"
                .logDebug()
        }

}