package com.instabug.crash;

import static com.instabug.commons.session.SessionIncident.ValidationStatus;
import static com.instabug.library.util.ReportHelper.getReport;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.anr.di.AnrServiceLocator;
import com.instabug.commons.caching.DiskHelper;
import com.instabug.commons.di.CommonsLocator;
import com.instabug.commons.threading.CrashDetailsParser;
import com.instabug.commons.threading.CrashDetailsParser.ErrorParsingStrategy;
import com.instabug.commons.threading.CrashDetailsParser.ThreadParsingStrategy;
import com.instabug.commons.utils.StateExtKt;
import com.instabug.crash.cache.CrashReportsDbHelper;
import com.instabug.crash.di.CrashesServiceLocator;
import com.instabug.crash.eventbus.NDKCrashReportingFeatureStateChange;
import com.instabug.crash.models.Crash;
import com.instabug.crash.models.IBGNonFatalException;
import com.instabug.crash.network.InstabugCrashesUploaderJob;
import com.instabug.crash.screenrecording.ExternalAutoScreenRecordHelper;
import com.instabug.crash.settings.PerSessionSettings;
import com.instabug.crash.utils.CrashEmailSetter;
import com.instabug.crash.utils.CrashReportingUtility;
import com.instabug.library.Feature;
import com.instabug.library.Instabug;
import com.instabug.library.Platform;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.AutoScreenRecordingEventBus;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation;
import com.instabug.library.internal.video.ScreenRecordingServiceAction;
import com.instabug.library.model.Report;
import com.instabug.library.model.State;
import com.instabug.library.model.UserAttributes;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.MD5Generator;
import com.instabug.library.util.ReportHelper;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;

/**
 * Created by tarek on 5/2/17.
 */

public class CrashReporting {
    private static final String TAG = "CrashReporting";
    public static final String CRASH_STATE = "crash_state";
    public static final String ANR_STATE = "anr_state";

    /**
     * Manually report handled errors and exceptions in your code.
     *
     * @param exception The non-fatal exception object to be reported
     * @see IBGNonFatalException
     */
    public static void report(@NonNull IBGNonFatalException exception) {
        APIChecker.checkAndRunInExecutor("CrashReporting.report", () ->
                reportHandledException(exception.getThrowable(), null, exception.getUserAttributes(), exception.getFingerprint(), exception.getLevel()));
    }

    private static void reportHandledException(@NonNull final Throwable throwable, @Nullable final String exceptionIdentifier,
                                               @Nullable final Map<String, String> userAttributes, @Nullable String fingerprint, @NonNull IBGNonFatalException.Level level) {
        if (throwable == null) return;
        InstabugSDKLogger.d(Constants.LOG_TAG, "Reporting handled exception: " + throwable.getMessage());
        if (isCrashTypeNotAllowedAndLog(true)) {
            return;
        }
        JSONObject formattedException = getFormattedException(throwable, exceptionIdentifier);
        if (formattedException == null) return;
        JSONObject formattedFingerprint = getFingerprintObject(fingerprint);
        if (formattedFingerprint == null)
            InstabugSDKLogger.e(Constants.LOG_TAG, Constants.FINGERPRINT_EMPTY_ERROR);
        reportException(formattedException, true, userAttributes, formattedFingerprint, level);
    }

    private static void reportException(@NonNull final JSONObject jsonObject, final boolean isHandled) {
        reportException(jsonObject, isHandled, null);
    }

    private static void reportException(@NonNull final JSONObject jsonObject, final boolean isHandled,
                                        @Nullable final Map<String, String> userAttributes) {
        reportException(jsonObject, isHandled, userAttributes, null);
    }

    private static void reportException(@NonNull final JSONObject jsonObject, final boolean isHandled,
                                        @Nullable final Map<String, String> userAttributes, @Nullable JSONObject fingerprint) {
        reportException(jsonObject, isHandled, userAttributes, fingerprint, IBGNonFatalException.Level.ERROR);
    }

    private static void reportException(@NonNull final JSONObject jsonObject, final boolean isHandled,
                                        @Nullable final Map<String, String> userAttributes, @Nullable JSONObject fingerprint, @NonNull IBGNonFatalException.Level level) {
        if (jsonObject == null) return;

        if (InstabugCore.getPlatform() != Platform.ANDROID && !isHandled) {
            reportCrashingSession();
        }
        if (!CrashReportingUtility.isCrashReportingEnabled()) {
            return;
        }
        if (isCrashTypeNotAllowedAndLog(isHandled)) {
            return;
        }
        String currentView = InstabugCore.getCurrentView();
        if (ExternalAutoScreenRecordHelper.getInstance().isEnabled()
                && SettingsManager.getInstance().isAutoScreenRecordingEnabled()) {
            deleteAutoScreenRecording();
        }
        if (isHandled || InstabugCore.getPlatform() == Platform.FLUTTER)
            PoolProvider.getSingleThreadExecutor("CRASHES")
                    .execute(() -> saveAndReportException(jsonObject, isHandled, userAttributes, fingerprint, level, currentView));
        else
            saveAndReportException(jsonObject, false, userAttributes, fingerprint, level, currentView);
    }

    private static boolean isCrashTypeNotAllowedAndLog(boolean isHandled) {
        if (isHandled && !CrashReportingUtility.isNonFatalReportingEnabled()) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "HandledExceptionReporting is disabled, Couldn't report error");
            return true;
        }
        return false;
    }

    private static void saveAndReportException(@NonNull JSONObject jsonObject, boolean isHandled, @Nullable Map<String, String> userAttributes, @Nullable JSONObject fingerprint, @NonNull IBGNonFatalException.Level level, String currentView) {
        Context context = Instabug.getApplicationContext();
        if (context == null) return;
        if (isHandled && CrashesServiceLocator.getIgnoreNonFatalValidator().validateIgnoreNonFatalResponse(jsonObject, fingerprint)) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Skipping this non fatal as it is filtered");
            return;

        }
        State state = State.getState(context);
        // Updating currentView of the created state with the currentView captured
        // before offloading the creation process to a background thread to avoid any delays.
        StateExtKt.updateScreenShotAnalytics(state);
        state.setCurrentView(currentView);
        CrashEmailSetter.updateStateEmailIfNeeded(state);
        Report report = getReport(InstabugCore.getOnReportCreatedListener());
        Crash crash = getCrash(jsonObject, isHandled, state, fingerprint, level, context);

        ReportHelper.update(crash.getState(), report);
        if (!isHandled) state.updateSessionIdFromLatestSession();
        if (userAttributes != null && !userAttributes.isEmpty()) {
            appendUserAttributes(state, userAttributes);
        }
        if (InstabugCore.getExtraAttachmentFiles() != null &&
                InstabugCore.getExtraAttachmentFiles().size() >= 1) {
            addCrashAttachments(context, crash);
        }

        File file = DiskHelper.getIncidentStateFile(crash.getSavingDirOnDisk(context), CRASH_STATE);
        createStateTextFile(context, crash, file);
        AttachmentsUtility.encryptAttachments(crash.getAttachments());

        CrashReportsDbHelper.trimAndInsert(crash);
        CommonsLocator.getSessionLinker().link(crash, ValidationStatus.VALIDATED);
        InstabugSDKLogger.d(Constants.LOG_TAG, "Crash " + crash.getMetadata().getUuid() + "is Linked successfully");

        InstabugSDKLogger.d(Constants.LOG_TAG, "Your exception has been reported");
        if (isHandled || InstabugCore.getPlatform() == Platform.FLUTTER)
            InstabugCrashesUploaderJob.getInstance().start();
        ExternalAutoScreenRecordHelper.getInstance().start();
    }

    /**
     * This method is not being used by the native SDK. Need to check if it's being used by CP.
     */
    @SuppressWarnings("unused")
    private static void createFormattedException(@NonNull Throwable throwable, @Nullable String identifier) {
        createFormattedException(throwable, identifier, null);
    }

    private static void createFormattedException(@NonNull Throwable throwable, @Nullable String identifier,
                                                 @Nullable Map<String, String> userAttributes) {
        if (throwable == null) return;

        InstabugSDKLogger.v(Constants.LOG_TAG, "Creating formatted exception for error: " +
                throwable.getClass().getCanonicalName());

        CrashDetailsParser parser = new CrashDetailsParser(
                ThreadParsingStrategy.None.INSTANCE,
                new ErrorParsingStrategy.Crashing(throwable, identifier)
        );
        reportException(parser.getCrashDetails(), true, userAttributes);
    }

    private static void appendUserAttributes(@NonNull State state, Map<String, String> userAttributesMap) {
        int userAttributeMaxLength = 90;
        int userAttributesMaxCount = 100;
        UserAttributes userAttributes = new UserAttributes();
        try {
            if (state.getUserAttributes() != null) {
                JSONObject jsonObject = new JSONObject(state.getUserAttributes());
                Iterator<String> iterator = jsonObject.keys();
                while (iterator.hasNext()) {
                    String key = (String) iterator.next();
                    String value = jsonObject.getString(key);
                    userAttributes.put(key, value);
                }
            }
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while appending user attributes to crash report", e);
        }
        if (userAttributesMap.size() > userAttributesMaxCount) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Some old user attributes were removed. " +
                    "Max allowed user attributes reached. " +
                    "Please note that you can add up to " + userAttributesMaxCount + " user attributes.");
            LinkedHashMap<String, String> userAttributesClone = new LinkedHashMap<>(userAttributesMap);
            for (Map.Entry<String, String> entry : userAttributesMap.entrySet()) {
                userAttributesClone.remove(entry.getKey());
                if (userAttributesClone.size() <= userAttributesMaxCount) {
                    break;
                }
            }
            userAttributesMap.clear();
            userAttributesMap.putAll(userAttributesClone);
        }

        for (Map.Entry<String, String> entry : userAttributesMap.entrySet()) {
            if (userAttributesMap.get(entry.getKey()) != null) {
                if (entry.getKey().length() > userAttributeMaxLength || entry.getValue().length() > userAttributeMaxLength) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "Some user attributes weren't added. Max allowed user attributes characters limit is reached. " +
                            "Please note that you can add user attributes (key, value) with characters count up to " + userAttributeMaxLength + " characters.");
                } else {
                    userAttributes.put(entry.getKey(), entry.getValue());
                }
            }
        }

        state.setUserAttributes(userAttributes.toString());

    }

    private static void reportCrashingSession() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Report crashing session");
        IBGCoreEventPublisher.post(IBGSdkCoreEvent.CrossPlatformCrashed.INSTANCE);
        PerSessionSettings.getInstance().setRNCrashedSession(true);
    }

    private static void createStateTextFile(@NonNull Context context, @NonNull Crash crash, @NonNull File file) {
        if (file == null || crash == null || crash.getState() == null) return;


        InstabugSDKLogger.v(Constants.LOG_TAG, "Creating state file for crash: " + crash.getId());
        try {
            Uri uri = DiskUtils.with(context)
                    .writeOperation(new WriteStateToFileDiskOperation(file, crash.getState().toJson()))
                    .execute();
            if (uri != null)
                crash.getState().setUri(uri);
        } catch (Throwable e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "error while creating state text file", e);
        }
    }

    public static void deleteAutoScreenRecording() {
        // TODO: 10/8/20: why public?
        AutoScreenRecordingEventBus.getInstance().post(ScreenRecordingServiceAction.CustomeActions.STOP_DELETE);
    }

    public static void addCrashAttachments(@NonNull Context context, Crash crash) {
        // TODO: 12/8/20 Why public?
        if (context == null || crash == null) return;

        if (InstabugCore.getExtraAttachmentFiles() != null) {
            for (Map.Entry<Uri, String> entry : InstabugCore.
                    getExtraAttachmentFiles().entrySet()) {
                Uri attachmentUri =
                        AttachmentsUtility.getNewFileAttachmentUri(context, entry.getKey(), entry
                                .getValue());
                if (attachmentUri != null) {
                    crash.addAttachment(attachmentUri);
                }
            }
        }
    }

    @Nullable
    @VisibleForTesting
    public static JSONObject getFormattedException(@NonNull Throwable throwable, @Nullable String identifier) {
        if (throwable == null) return null;

        InstabugSDKLogger.v(Constants.LOG_TAG, "Creating formatted exception for error: " +
                throwable.getClass().getCanonicalName());

        CrashDetailsParser parser = new CrashDetailsParser(
                ThreadParsingStrategy.None.INSTANCE,
                new ErrorParsingStrategy.Crashing(throwable, identifier)
        );
        JSONObject crashDetails = parser.getCrashDetails();
        return crashDetails.length() != 0 ? crashDetails : null;
    }

    @Nullable
    @VisibleForTesting
    public static JSONObject getFingerprintObject(@Nullable String fingerprint) {
        if (fingerprint == null) return null;
        String operableFingerprint = fingerprint.trim();
        if (operableFingerprint.length() == 0) return null;
        try {
            JSONObject fingerprintJson = new JSONObject();
            String fingerprintMD5 = MD5Generator.generateMD5(operableFingerprint.toLowerCase(Locale.getDefault()));
            if (fingerprintMD5 == null || fingerprintMD5.isEmpty())
                throw new IllegalStateException("Couldn't generate MD5 for fingerprint");
            fingerprintJson.put("md5", fingerprintMD5);
            fingerprintJson.put("length", operableFingerprint.length());
            if (operableFingerprint.length() > 40) {
                InstabugSDKLogger.w(Constants.LOG_TAG, Constants.FINGERPRINT_LIMIT_WARNING);
                operableFingerprint = operableFingerprint.substring(0, 40);
            }
            fingerprintJson.put("original", operableFingerprint);
            return fingerprintJson;
        } catch (Throwable throwable) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to process fingerprint", throwable);
            return null;
        }
    }

    @NonNull
    private static Crash getCrash(JSONObject jsonObject, boolean handled, State state, @Nullable JSONObject fingerprint, @NonNull IBGNonFatalException.Level level, @NonNull Context context) {
        Crash crash = new Crash.Factory().create(state, context, handled);
        crash.setCrashMessage(jsonObject.toString());
        crash.setCrashState(Crash.CrashState.READY_TO_BE_SENT);
        crash.setHandled(handled);
        crash.setLevel(level);
        CrashDetailsParser parser = new CrashDetailsParser(
                ThreadParsingStrategy.None.INSTANCE,
                ErrorParsingStrategy.None.INSTANCE
        );
        crash.setThreadsDetails(parser.getThreadsDetails().toString());
        crash.setFingerprint(fingerprint != null ? fingerprint.toString() : null);
        return crash;
    }

    /**
     * This API is Created for cross platform plugins usage.
     */
    private static void reportUncaughtException(@NonNull JSONObject jsonObject) {
        if (jsonObject == null) return;

        reportException(jsonObject, false);
    }

    /**
     * Enable/disable crash reporting
     *
     * @param state desired state of crash reporting feature
     * @see com.instabug.library.Feature.State
     */
    public static void setState(@NonNull final Feature.State state) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "CrashReporting setState:" + state);
        if (state == Feature.State.DISABLED)
            CrashesServiceLocator
                    .getCrashConfigurationProvider()
                    .setCrashReportingLocallyEnabled(false);
        APIChecker.checkAndRunInExecutor("CrashReporting.setState", new VoidRunnable() {
            @Override
            public void run() {
                if (state == Feature.State.ENABLED && !CrashesServiceLocator.getCrashConfigurationProvider().isCrashReportingAvailable()) {
                    InstabugSDKLogger.e("Instabug-CrashReporting", "crash reporting " +
                            "wasn't enabled as it seems to be disabled for your Instabug company account. " +
                            "Please, contact support to switch it on for you.");
                    return;
                }
                CrashesServiceLocator.getCrashConfigurationProvider().setCrashReportingLocallyEnabled(state == Feature.State.ENABLED);
                IBGCoreEventPublisher.post(Features.Updated.INSTANCE);
            }
        });
    }

    /**
     * Enable/disable ANR reporting
     *
     * @param state desired state of ANR reporting feature
     * @throws IllegalStateException if Instabug object wasn't built using {@link
     *                               Instabug.Builder#build()} before this method was called
     * @see com.instabug.library.Feature.State
     */
    public static void setAnrState(@NonNull final Feature.State state) {
        if (state == Feature.State.DISABLED)
            AnrServiceLocator
                    .getAnrConfigurationProvider()
                    .setAnrLocallyEnabled(false);
        APIChecker.checkAndRunInExecutor("CrashReporting.setAnrState", new VoidRunnable() {
            @Override
            public void run() {
                if (state == Feature.State.ENABLED && !CrashReportingUtility.isCrashReportingEnabled()) {
                    // User wants to enable ANR reporting while Crash reporting is disabled
                    InstabugSDKLogger.w(Constants.LOG_TAG, "Can not enable ANR reporting while Crash reporting is disabled");
                    return;
                }
                AnrServiceLocator.getAnrConfigurationProvider().setAnrLocallyEnabled(state == Feature.State.ENABLED);
                IBGCoreEventPublisher.post(Features.Updated.INSTANCE);
            }
        });
    }

    /**
     * Enable/disable ndk crash reporting
     *
     * @param state desired state of crash reporting feature
     * @see com.instabug.library.Feature.State
     */
    public static void setNDKCrashesState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("CrashReporting.setNDKCrashesState", new VoidRunnable() {
            @Override
            public void run() {
                if (NDKCrashReportingFeatureStateChange.getInstance() != null) {
                    NDKCrashReportingFeatureStateChange.getInstance().post(state);
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't not enable NDK crash reporting state is null.");
                }
            }
        });
    }

    /**
     * sets onCrashSentCallback {@link OnCrashSentCallback}
     *
     * @param onCrashSentCallback a callback that will be invoked when when crash is sent to dashboard
     */
    public static void setOnCrashSentCallback(@NonNull final OnCrashSentCallback onCrashSentCallback) {
        //setting the callback before any check on user thread
        // to prevent any race-condition when sending crashes on launch
        CommonsLocator.setUserCrashMetadataCallback(onCrashSentCallback);
        APIChecker.checkAndRunInExecutor("CrashReporting.setOnCrashSentCallback",
                () -> {
                    if (!CrashesServiceLocator.getCrashConfigurationProvider().isCrashReportingEnabled())
                        InstabugSDKLogger.e(Constants.LOG_TAG, "CrashReporting.setOnCrashSentCallback Call won’t take effect as CrashReporting is disabled. Please enable CrashReporting first.");
                });
    }

    /**
     * Enable/Disable sending user identification data (username & email) set by calling
     * {@link Instabug#identifyUser(String, String, String)} with any crash report
     *
     * @param state desired state of user identification
     */
    public static void setUserIdentificationState(Feature.State state) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "CrashReporting.setUserIdentificationState() called with state" + (state == Feature.State.ENABLED ? "Enabled" : "Disabled"));
        CommonsLocator.getConfigurationsProvider().setUserIdentificationEnabled(state == Feature.State.ENABLED);
    }
}
