package com.instabug.crash

import android.content.Context
import com.instabug.commons.PluginDelegate
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.di.CommonsLocator.captorsRegistry
import com.instabug.commons.di.CommonsLocator.configurationsHandler
import com.instabug.commons.di.CommonsLocator.configurationsProvider
import com.instabug.commons.di.CommonsLocator.crashesCacheDir
import com.instabug.commons.di.CommonsLocator.crashesSessionDataController
import com.instabug.commons.di.CommonsLocator.reproProxy
import com.instabug.commons.di.CommonsLocator.sessionLinker
import com.instabug.commons.snapshot.StateSnapshotCaptor
import com.instabug.crash.di.CrashesServiceLocator.nonFatalsConfigurationsProvider
import com.instabug.crash.di.CrashesServiceLocator.nonFatalsReproConfigurationsHandler
import com.instabug.crash.di.CrashesServiceLocator.pluginDelegates
import com.instabug.crash.settings.CrashSettings
import com.instabug.crash.utils.CrashReportingUtility.isCrashReportingEnabled
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.eventpublisher.IBGCompositeDisposable
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.core.plugin.Plugin
import com.instabug.library.screenshot.analytics.AnalyticsEvent
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus
import com.instabug.library.sessionV3.providers.FeatureSessionDataController
import com.instabug.library.sessionV3.providers.FeatureSessionDataControllerHost
import com.instabug.library.sessionV3.providers.FeatureSessionLazyDataProvider
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider

class CrashPlugin : Plugin(), FeatureSessionDataControllerHost, FeatureSessionLazyDataProvider {
    @Volatile
    var isLastEnabled = true

    val disposables by lazy { IBGCompositeDisposable() }
    var screenShotAnalyticsDisposables: IBGDisposable? = null
    override fun init(context: Context) {
        super.init(context)
        reproProxy.evaluate(configurationsProvider)
        onDelegates { init(context) }

    }

    override fun start(context: Context) {
        SettingsManager.getInstance()
            ?.reproConfigurations
            ?.let {
                PoolProvider.postIOTaskWithCheck { handleReproStateConfigurations(it.modesMap) }
            }
        onDelegates { start(context) }
        isLastEnabled = isCrashReportingEnabled()
        PoolProvider.postIOTaskWithCheck(this::subscribeOnSDKEvents)
        subscribeToProductAnalyticsCollector()
    }

    override fun wake() {
        if (isLastEnabled) {
            crashesCacheDir.setCurrentSessionId(InstabugCore.getRunningSession()?.id)
            captorsRegistry.forceOnAvailable(StateSnapshotCaptor.ID)
        }
        onDelegates(PluginDelegate::wake)
    }

    override fun sleep() {
        onDelegates(PluginDelegate::sleep)
    }

    override fun stop() {
        crashesCacheDir.setCurrentSessionId(null)
        onDelegates(PluginDelegate::stop)
        disposables.dispose()
        unSubscribeFromProductAnalyticsCollector()
    }

    override fun getLastActivityTime(): Long = CrashSettings.getInstance().lastCrashTime

    override fun isFeatureEnabled(): Boolean = isCrashReportingEnabled()

    override val sessionDataController: FeatureSessionDataController
        get() = crashesSessionDataController

    override fun isDataReady(sessionIds: List<String>): Map<String, Boolean> =
        sessionLinker.inspectDataReadiness(sessionIds)

    private fun onDelegates(action: PluginDelegate.() -> Unit) {
        pluginDelegates.forEach(action)
    }

    fun subscribeOnSDKEvents() {
        disposables.add(subscribeToIBGCoreEvents())
    }

    private fun subscribeToIBGCoreEvents() = IBGCoreEventSubscriber.subscribe { event ->
        onDelegates { handleSDKCoreEvent(event) }
        handleSDKCoreEvent(event)
    }

    private fun handleSDKCoreEvent(event: IBGSdkCoreEvent) {
        when (event) {
            is IBGSdkCoreEvent.Features -> handleStateChange()
            is IBGSdkCoreEvent.FeaturesFetched -> {
                configurationsHandler.handleConfiguration(event.response)
                handleStateChange()
            }

            is IBGSdkCoreEvent.ReproState -> handleReproStateConfigurations(event.modesMap)
            is IBGSdkCoreEvent.V3Session.V3StartedInForeground ->
                captorsRegistry.forceOnAvailable(StateSnapshotCaptor.ID)

            else -> return
        }
    }

    private fun handleReproStateConfigurations(modesMap: Map<Int, Int>) {
        configurationsHandler.handle(modesMap)
        reproProxy.evaluate(configurationsProvider)
        ScreenshotsAnalyticsEventBus.post(AnalyticsEvent.ScreenshotEvent.OnConfigurationChanged)
    }

    private fun handleStateChange() {
        reproProxy.evaluate(configurationsProvider)
        ScreenshotsAnalyticsEventBus.post(AnalyticsEvent.ScreenshotEvent.OnConfigurationChanged)
        if (isLastEnabled == isCrashReportingEnabled()) return
        if (isCrashReportingEnabled()) {
            crashesCacheDir.setCurrentSessionId(InstabugCore.getRunningSession()?.id)
            captorsRegistry.forceOnAvailable(StateSnapshotCaptor.ID)
            isLastEnabled = true
            return
        }
        with(crashesCacheDir) { setCurrentSessionId(null); deleteFileDir() }
        isLastEnabled = false
    }

    private fun subscribeToProductAnalyticsCollector() {
        if (screenShotAnalyticsDisposables == null) {
            screenShotAnalyticsDisposables = ScreenshotsAnalyticsEventBus.subscribe(
                CommonsLocator.crashReportingProductAnalyticsCollector
            )
        }
    }


    private fun unSubscribeFromProductAnalyticsCollector() {
        screenShotAnalyticsDisposables?.dispose()
        screenShotAnalyticsDisposables = null
    }

}
