package com.instabug.commons.diagnostics.event

import android.app.ApplicationExitInfo
import androidx.annotation.StringDef
import com.instabug.anr.diagnostics.ANRIncidentType
import com.instabug.commons.diagnostics.di.DiagnosticsLocator
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent.Action
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent.IncidentType
import com.instabug.crash.diagnostics.CrashIncidentType
import com.instabug.terminations.diagnostics.TerminationIncidentType


private const val CALIBRATION_DIAGNOSTIC_EVENT_DEFAULT_COUNT = 1

private const val CALIBRATION_DIAGNOSTIC_EVENT_KEY = "%s_%s_%s"

private const val CALIBRATION_DIAGNOSTIC_EVENT_STRING_FORMAT = "(key -> %s, count -> %d)"

/**
 * A [DiagnosticEvent] that should be reported for SDK incident capturing and OS recoreded exit
 * reasons for comparison and accuracy calibration.
 *
 * @constructor Creates and event with [IncidentType] and [Action]
 * @param incidentType the type of incident that's being reported
 * @param action the action that's being reported
 */
class CalibrationDiagnosticEvent(
    incidentType: IncidentType,
    @Action action: String,
    @Source source: String
) : DiagnosticEvent {

    constructor(incidentType: IncidentType, action: String) : this(incidentType, action, Source.SDK)

    @JvmOverloads
    constructor(
        exitReason: Int,
        mapper: OSExitReasonMapper = DefaultOSExitReasonMapper()
    ) : this(mapper.map(exitReason), Action.Captured, Source.OS)

    override val key: String = CALIBRATION_DIAGNOSTIC_EVENT_KEY
        .format(incidentType.diagnosticsName, source, action).lowercase()

    override val count: Int = CALIBRATION_DIAGNOSTIC_EVENT_DEFAULT_COUNT

    override val reportingPredicate: () -> Boolean = incidentType.reportingPredicate

    override fun toString(): String = CALIBRATION_DIAGNOSTIC_EVENT_STRING_FORMAT.format(key, count)

    interface IncidentType {
        val diagnosticsName: String
        val reportingPredicate: () -> Boolean
    }

    @Retention(AnnotationRetention.RUNTIME)
    @StringDef(Action.Captured, Action.Synced)
    annotation class Action {
        companion object {
            const val Captured = "captured"
            const val Synced = "synced"
        }
    }


    @Retention(AnnotationRetention.RUNTIME)
    @StringDef(Source.SDK, Source.OS)
    annotation class Source {
        companion object {
            const val SDK = "sdk"
            const val OS = "os"
        }
    }

    /**
     * Base mapper to transform OS's exit reason meta to [IncidentType]
     */
    interface OSExitReasonMapper {
        /**
         * Maps exit reason to [IncidentType]
         * @param exitReason The retrieved OS's exit reason.
         */
        fun map(exitReason: Int): IncidentType
    }

    class DefaultOSExitReasonMapper : OSExitReasonMapper {
        override fun map(exitReason: Int): IncidentType = when (exitReason) {
            ApplicationExitInfo.REASON_CRASH -> CrashIncidentType()
            ApplicationExitInfo.REASON_CRASH_NATIVE -> DiagnosticsLocator.ndkIncidentTypeGetter()
            ApplicationExitInfo.REASON_ANR -> ANRIncidentType()
            ApplicationExitInfo.REASON_USER_REQUESTED -> TerminationIncidentType()
            else -> object : IncidentType {
                override val diagnosticsName: String = "unknown"
                override val reportingPredicate: () -> Boolean = { false }
            }
        }
    }
}