package com.instabug.early_crash.network

import com.instabug.commons.logging.logError
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.featuresflags.HelperMethods
import com.instabug.library.featuresflags.constants.FFMode
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator
import com.instabug.library.featuresflags.mappers.featuresFlagsAsJson
import com.instabug.library.model.State
import com.instabug.library.networkv2.request.AppTokenProvider
import com.instabug.library.networkv2.request.Endpoints
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestMethod
import com.instabug.library.settings.SettingsManager
import org.json.JSONArray
import org.json.JSONObject

class EarlyCrashRequestFactory(
    private val settingsManager: SettingsManager
) : ParameterizedFactory<Request?, JSONObject> {

    override fun create(type: JSONObject): Request? =
        type.createTokenProvider()?.let { appTokenProvider ->
            type.updateFeatureFlags()
            Request.Builder()
                .endpoint(Endpoints.REPORT_CRASH)
                .method(RequestMethod.POST)
                .tokenProvider(appTokenProvider)
                .setRequestBodyJsonObject(type)
                .build()
        }

    private fun JSONObject.createTokenProvider(): AppTokenProvider? =
        (getNullableString(TOKEN_KEY) ?: settingsManager.earlyAppToken)
            ?.let(::createTokenProvider)
            ?: let { "Early crash request factory cannot resolve Instabug SDK app token".logError(); null }

    private fun createTokenProvider(token: String): AppTokenProvider = object : AppTokenProvider {
        override fun getAppToken(): String = token
    }

    private fun JSONObject.getNullableString(key: String): String? =
        runCatching { getString(key) }.getOrNull()

    private fun JSONObject.updateFeatureFlags() {
        runCatching {
            val ffMode = FeaturesFlagServiceLocator.featuresFlagsConfigsProvider.mode
            when (ffMode) {
                FFMode.DISABLED -> {
                    this.remove(State.KEY_FEATURES_FLAGS)
                }

                FFMode.FF -> {
                    val featuresFlagsAsObject =
                        featuresFlagsAsJson(this.getJSONArray(State.KEY_FEATURES_FLAGS))
                    this.put(State.KEY_FEATURES_FLAGS, featuresFlagsAsObject)
                }

                FFMode.EXPERIMENTS -> {
                    val featureFlagsJsonArray = try {
                        this.getJSONArray(State.KEY_FEATURES_FLAGS)
                    } catch (t: Throwable) {
                        JSONArray()
                    }
                    val experimentsJsonArray: JSONArray = try {
                        this.getJSONArray(State.KEY_EXPERIMENTS)
                    } catch (t: Throwable) {
                        JSONArray()
                    }
                    for (i in 0 until featureFlagsJsonArray.length()) {
                        experimentsJsonArray.put(
                            HelperMethods.formatFeaturesFlagList(
                                featureFlagsJsonArray.getJSONObject(i)
                            )
                        )
                    }
                    this.put(State.KEY_EXPERIMENTS, experimentsJsonArray)
                    this.remove(State.KEY_FEATURES_FLAGS)
                }

                else -> {/*Do Nothing*/
                }
            }
        }
    }

    companion object {
        private const val TOKEN_KEY = State.KEY_APP_TOKEN
    }
}