package com.instabug.crash.utils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.commons.di.CommonsLocator;
import com.instabug.crash.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

public class ExceptionFormatter {

    public static JSONObject createExceptionJson(@NonNull Throwable throwable, @Nullable String identifier) {
        JSONObject exception = new JSONObject();

        if (throwable == null) return exception;

        try {
            String name = throwable.getClass().getName();
            if (identifier != null) {
                name += "-" + identifier;
            }
            exception.put("name", name);
            StackTraceElement element = null;
            if (throwable.getStackTrace() != null && throwable.getStackTrace().length > 0) {
                element = throwable.getStackTrace()[0];
            }

            if (element != null && element.getFileName() != null) {
                exception.put("location", element.getFileName() + ":" + element.getLineNumber());
            } else {
                InstabugSDKLogger.w(Constants.LOG_TAG, "Incomplete crash stacktrace, if you're using Proguard, add the following line to your configuration file to have file name and line number in your crash report:");
                InstabugSDKLogger.w(Constants.LOG_TAG, "-keepattributes SourceFile,LineNumberTable");
            }

            exception.put("exception", throwable.toString());
            if (throwable.getMessage() != null) exception.put("message", throwable.getMessage());
            exception.put("stackTrace", throwableStackTrace(throwable));
            if (throwable.getCause() != null) {
                exception.put("cause", createExceptionJson(throwable.getCause(), identifier));
            }
        } catch (JSONException jsonException) {
            InstabugSDKLogger.e(Constants.LOG_TAG, jsonException.getMessage() != null
                    ? jsonException.getMessage()
                    : "Json exception while creating formatted exception", jsonException);
        }
        return exception;
    }

    public static String throwableStackTrace(@NonNull Throwable throwable) {
        StringBuilder trace = new StringBuilder(throwable.toString()).append("\n");
        StackTraceElement[] stackTrace = throwable.getStackTrace();
        int maxFrames = Math.min(stackTrace.length, CommonsLocator.getThreadingLimitsProvider().provideErrorThreadFramesLimit());
        for (int index = 0; index < maxFrames; index++) {
            trace.append("\t at ").append(stackTrace[index].toString()).append("\n");
        }
        return trace.toString();
    }
}
