package com.instabug.commons.utils

import com.instabug.commons.caching.getOldSpanDir
import com.instabug.commons.di.CommonsLocator
import com.instabug.library.IssueType
import com.instabug.library.ReproMode
import com.instabug.library.SpansCacheDirectory
import com.instabug.library.datahub.HubReportModifier
import com.instabug.library.internal.filestore.MatchingIDSpanSelector
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.model.State
import com.instabug.library.util.extenstions.toMap
import org.json.JSONObject
import java.io.File

fun State?.modifyWithHubData() {
    val associatedLaunchId = this?.appLaunchId ?: return
    HubReportModifier
        .Builder()
        .buildWithDefaultStores()
        .also { modifier -> modifier.prepare(this, MatchingIDSpanSelector(associatedLaunchId)) }
        .also { modifier -> modifier.finish() }
}

fun State?.dropReproStepsIfNeeded(issueType: @IssueType Int) =
    this
        ?.reproConfigurationsMap
        ?.let { map ->
            map[issueType]
        }?.takeIf { mode ->
            mode == ReproMode.Disable
        }?.let {
            setVisualUserSteps(null)
        }

fun State?.getScreenshotsDir(
    reproScreenshotsDir: SpansCacheDirectory,
    issueType: @IssueType Int,
): File? =
    this
        ?.reproConfigurationsMap
        ?.let { map ->
            map[issueType]
        }?.takeIf { mode ->
            mode > ReproMode.EnableWithNoScreenshots
        }?.let {
            reproScreenshotsDir.getOldSpanDir(this)
        }

fun State?.generateReproConfigsMap() {
    val reproMap: HashMap<Int, Int> = hashMapOf<Int, Int>()
    val types = listOf(IssueType.Fatal, IssueType.ANR, IssueType.ForceRestart)
    types.forEach { type ->
        val configProvider = getReproConfigurationProviderForCrashType(type)
        reproMap[type] = configProvider.getReproMode()
    }
    this?.reproConfigurationsMap = reproMap
}

fun State?.updateScreenShotAnalytics() {
    this?.screenShotAnalytics =
        CoreServiceLocator.screenShotAnalyticsMapper
            .map(CommonsLocator.crashReportingProductAnalyticsCollector.getAnalytics())
            .toString()
}

fun State?.toUserAttributesMap() =
    runCatching {
        this
            ?.userAttributes
            ?.let(::JSONObject)
            ?.toMap()
            ?.filterValues { value -> value != null }
            ?.mapValues { attribute -> attribute.value.toString() }
    }.getOrNull()
