package com.instabug.terminations.sync

import com.instabug.commons.diagnostics.di.DiagnosticsLocator
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent.Action
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.logging.logWarning
import com.instabug.crash.settings.CrashSettings
import com.instabug.crash.utils.deleteAttachment
import com.instabug.library.IBGNetworkWorker
import com.instabug.library.InstabugNetworkJob
import com.instabug.library.core.InstabugCore
import com.instabug.library.frustratingexperience.FrustratingExperienceEvent
import com.instabug.library.frustratingexperience.FrustratingExperienceEventBus
import com.instabug.library.internal.storage.AttachmentsUtility
import com.instabug.library.model.Attachment
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.limitation.RateLimiter
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.terminations.Constants
import com.instabug.terminations.di.ServiceLocator
import com.instabug.terminations.diagnostics.TerminationIncidentType
import com.instabug.terminations.model.Termination
import com.instabug.terminations.model.TerminationState
import org.json.JSONObject
import java.io.File

private const val WARNING_DECRYPTION_FAILED =
    "Skipping Attachment file of type %s because it was not decrypted successfully."
private const val WARNING_FILE_NOT_EXISTS =
    "Skipping attachment file of type %s because it's either not found or empty."

class TerminationsSyncJob : InstabugNetworkJob() {

    private val networkManager: NetworkManager by lazy { ServiceLocator.networkManager }

    private val rateLimiter: RateLimiter<Termination, CrashSettings> by lazy {
        ServiceLocator.getRateLimiter { termination ->
            termination.clearStateFile()
            with(ServiceLocator) { appCtx?.let { ctx -> cachingManager.delete(ctx, termination) } }
        }
    }

    override fun start() {
        enqueueJob(IBGNetworkWorker.CRASH) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Starting terminations sync job")
            with(ServiceLocator) {
                val ctx = appCtx
                if (ctx != null) {
                    cachingManager.retrieve(ctx)
                        .filter { termination -> termination.incidentState > 0 }
                        .forEach { termination -> report(termination.apply { readStateFile(ctx) }) }
                }
            }
        }
    }

    private fun report(termination: Termination) {
        if (termination.incidentState != TerminationState.READY_TO_BE_SENT)
            return uploadLogs(termination)
        if (rateLimiter.applyIfPossible(termination)) return
        val request = TerminationRequestBuilder().reportRequest(termination)
        val callbacks = object : Request.Callbacks<RequestResponse, Throwable> {
            override fun onSucceeded(response: RequestResponse?) {
                rateLimiter.reset()
                val crashId = response?.responseBody?.let { body ->
                    kotlin.runCatching { JSONObject(body as String).getString("id") }
                        .getOrReport(null, "Failed to extract crash id")
                        .also { crashId ->
                            FrustratingExperienceEventBus.post(
                                FrustratingExperienceEvent.Synced(
                                    termination.id,
                                    crashId
                                )
                            )
                        }
                } ?: return
                termination.apply {
                    temporaryServerToken = crashId
                    incidentState = TerminationState.LOGS_READY_TO_BE_UPLOADED
                }
                ServiceLocator.cachingManager.update(termination)
                uploadLogs(termination)
            }

            override fun onFailed(error: Throwable?) {
                if (error == null) return
                if (rateLimiter.inspect(error, termination)) return
                InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to report termination", error)
            }
        }
        InstabugSDKLogger.d(Constants.LOG_TAG, "Reporting termination ${termination.id}")
        networkManager.doRequestOnSameThread(RequestType.NORMAL, request, callbacks)
    }

    private fun uploadLogs(termination: Termination) {
        if (termination.incidentState != TerminationState.LOGS_READY_TO_BE_UPLOADED)
            return uploadAttachments(termination)
        val request = TerminationRequestBuilder().uploadLogsRequest(termination)
        val callbacks = object : Request.Callbacks<RequestResponse, Throwable> {
            override fun onSucceeded(response: RequestResponse?) {
                termination.incidentState = TerminationState.ATTACHMENTS_READY_TO_BE_SENT
                ServiceLocator.cachingManager.update(termination)
                uploadAttachments(termination)
                DiagnosticsLocator.reporter
                    .report(CalibrationDiagnosticEvent(TerminationIncidentType(), Action.Synced))
            }

            override fun onFailed(error: Throwable?) {
                if (error == null) return
                InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to upload termination logs", error)
            }
        }
        InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading logs for termination ${termination.id}")
        networkManager.doRequestOnSameThread(RequestType.NORMAL, request, callbacks)
    }

    private fun uploadAttachments(termination: Termination) {
        if (termination.incidentState != TerminationState.ATTACHMENTS_READY_TO_BE_SENT)
            return delete(termination)
        var syncedCount = 0
        val finalSuccessCallback = object : Request.Callbacks<Attachment, Throwable> {
            override fun onSucceeded(response: Attachment?) {
                syncedCount++
                response?.let { deleteAttachment(it, termination.id.toString()) }
                if (syncedCount < termination.attachments.size) return
                termination.incidentState = TerminationState.SYNCHRONIZED
                ServiceLocator.cachingManager.update(termination)
                delete(termination)
            }

            override fun onFailed(error: Throwable?) {
                "Uploading terminations attachments failed".logVerbose()
                AttachmentsUtility.encryptAttachments(termination.attachments)
            }
        }
        termination.attachments.takeUnless { attachments -> attachments.isEmpty() }
            ?.asSequence()
            ?.filter(this::decryptOrLog)
            ?.filter(this::attachmentFileExistsOrLog)
            ?.map { attachment -> attachment to attachmentRequest(termination, attachment) }
            ?.filter { (_, request) -> request != null }
            ?.forEach { (attachment, request) ->
                fireAttachmentRequest(attachment, requireNotNull(request), finalSuccessCallback)
            } ?: run {
            termination.incidentState = TerminationState.SYNCHRONIZED
            ServiceLocator.cachingManager.update(termination)
            delete(termination)
        }
    }

    private fun decryptOrLog(attachment: Attachment): Boolean {
        if (AttachmentsUtility.decryptAttachmentAndUpdateDb(attachment)) return true
        WARNING_DECRYPTION_FAILED.format(attachment.type).logWarning()
        return false
    }

    private fun attachmentFileExistsOrLog(attachment: Attachment): Boolean =
        attachment.localPath?.let(::File)
            ?.takeIf(File::exists)
            ?.takeIf { file -> file.length() > 0 }
            ?.let { true }
            ?: run { WARNING_FILE_NOT_EXISTS.format(attachment.type).logWarning(); false }

    private fun attachmentRequest(termination: Termination, attachment: Attachment) =
        TerminationRequestBuilder().singleAttachmentRequest(termination, attachment)

    private fun fireAttachmentRequest(
        attachment: Attachment,
        request: Request,
        callback: Request.Callbacks<Attachment, Throwable>
    ) {
        val singleAttachmentCallback = object : Request.Callbacks<RequestResponse, Throwable> {
            override fun onSucceeded(response: RequestResponse?) {
                "Uploading termination attachment succeeded".logVerbose()
                callback.onSucceeded(attachment)
            }

            override fun onFailed(error: Throwable?) {
                "Uploading termination attachment failed with error ${error?.message}".logVerbose()
                callback.onFailed(error)
            }

        }
        networkManager.doRequestOnSameThread(
            RequestType.MULTI_PART,
            request,
            singleAttachmentCallback
        )
    }

    private fun delete(termination: Termination) {
        if (termination.incidentState != TerminationState.SYNCHRONIZED) return
        with(ServiceLocator) { appCtx?.let { ctx -> cachingManager.delete(ctx, termination) } }
        termination.clearStateFile()
        ServiceLocator.appCtx?.let(termination::getSavingDirOnDisk)
            ?.takeIf { it.exists() }
            ?.deleteRecursively()
    }

    private fun <R> Result<R>.getOrReport(default: R, message: String) = getOrElse { throwable ->
        InstabugSDKLogger.e(Constants.LOG_TAG, message, throwable)
        InstabugCore.reportError(throwable, message)
        default
    }
}