package com.instabug.early_crash.caching

import com.instabug.early_crash.threading.ExecutionMode
import com.instabug.library.internal.filestore.CreateNewFile
import com.instabug.library.internal.filestore.DeleteFile
import com.instabug.library.internal.filestore.DeleteOldestFilesOnLimit
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.FileSelector
import com.instabug.library.internal.filestore.JSONObjectAggregator
import com.instabug.library.internal.filestore.MostRecentFileSelector
import com.instabug.library.internal.filestore.OperationScopeBuilder
import com.instabug.library.internal.filestore.ReadJSONFromFile
import com.instabug.library.internal.filestore.WriteJSONToFile
import com.instabug.library.util.extenstions.deleteRecursivelyDefensive
import com.instabug.library.util.extenstions.ifNotExists
import com.instabug.library.util.extenstions.mkdirsDefensive
import com.instabug.library.util.extenstions.takeIfExists
import org.json.JSONObject

interface IEarlyCrashCacheHandler {
    fun save(id: Long, crashJson: JSONObject, execMode: ExecutionMode = ExecutionMode.Immediate)
    fun getIds(execMode: ExecutionMode = ExecutionMode.Immediate): List<String>
    fun get(id: String, execMode: ExecutionMode = ExecutionMode.Immediate): JSONObject?
    fun getMostRecent(execMode: ExecutionMode = ExecutionMode.Immediate): Pair<String, JSONObject>?
    fun delete(id: String, execMode: ExecutionMode = ExecutionMode.Immediate)
    fun trim(limit: Int, execMode: ExecutionMode = ExecutionMode.Immediate)
    fun clearAll(execMode: ExecutionMode = ExecutionMode.Immediate)
}

class EarlyCrashCacheHandler(
    private val directoryFactory: ReportsDirectory.Factory
) : IEarlyCrashCacheHandler {

    private var reportsDirectory: ReportsDirectory? = null

    override fun save(id: Long, crashJson: JSONObject, execMode: ExecutionMode) {
        execMode.invoke {
            onReportsDir { dir ->
                dir.ifNotExists { mkdirsDefensive() }
                val fileSelector = getFileSelector(id.toString())
                OperationScopeBuilder(CreateNewFile(fileSelector))
                    .continueWith(WriteJSONToFile(fileSelector, crashJson))
                    .buildAndExecute(dir)
            }
        }
    }

    override fun getIds(execMode: ExecutionMode): List<String> = execMode.invoke<List<String>> {
        onActiveReportsDir { dir ->
            dir.listFiles()?.map { file -> file.nameWithoutExtension }
        }.orEmpty()
    }

    override fun get(id: String, execMode: ExecutionMode): JSONObject? =
        execMode.invoke<JSONObject?> {
            onActiveReportsDir {
                ReadJSONFromFile(
                    getFileSelector(id),
                    JSONObjectAggregator()
                ).invoke(it)
            }
        }

    override fun getMostRecent(execMode: ExecutionMode): Pair<String, JSONObject>? =
        execMode.invoke<Pair<String, JSONObject>?> {
            onActiveReportsDir {
                MostRecentFileSelector<Directory>().invoke(it)?.let { file ->
                    ReadJSONFromFile(
                        FileSelector { file },
                        JSONObjectAggregator()
                    ).invoke(it)?.let { jsonObject ->
                        Pair(file.nameWithoutExtension, jsonObject)
                    }
                }
            }
        }

    override fun delete(id: String, execMode: ExecutionMode) {
        execMode.invoke {
            onActiveReportsDir { DeleteFile(getFileSelector(id)).invoke(it) }
        }
    }

    override fun trim(limit: Int, execMode: ExecutionMode) {
        execMode.invoke {
            onActiveReportsDir { DeleteOldestFilesOnLimit<ReportsDirectory>(limit).invoke(it) }
        }
    }

    override fun clearAll(execMode: ExecutionMode) {
        execMode.invoke {
            onActiveReportsDir { dir -> dir.deleteRecursivelyDefensive() }
        }
    }

    private fun getFileSelector(id: String) =
        FileSelector<ReportsDirectory> { onActiveReportsDir { dir -> dir.reportFile(id) } }

    private inline fun <Out> onActiveReportsDir(op: (ReportsDirectory) -> Out?): Out? {
        if (reportsDirectory == null)
            reportsDirectory = directoryFactory()
        return reportsDirectory?.takeIfExists()?.let(op)
    }

    private inline fun <Out> onReportsDir(op: (ReportsDirectory) -> Out?): Out? {
        if (reportsDirectory == null)
            reportsDirectory = directoryFactory()
        return reportsDirectory?.let(op)
    }
}