package com.instabug.crash

import android.content.Context
import androidx.annotation.WorkerThread
import com.instabug.commons.PluginDelegate
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.di.CommonsLocator
import com.instabug.crash.cache.CrashReportsDbHelper
import com.instabug.crash.configurations.NonFatalsConfigurationProvider
import com.instabug.crash.configurations.NonFatalsReproConfigurationHandler
import com.instabug.crash.di.CrashesServiceLocator
import com.instabug.crash.network.InstabugCrashesUploaderJob
import com.instabug.crash.settings.CrashSettings
import com.instabug.crash.settings.PersistableSettings
import com.instabug.crash.utils.CrashCleanupUtils
import com.instabug.crash.utils.CrashReportingUtility.isCrashReportingEnabled
import com.instabug.library.Instabug
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.NetworkActivated
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.User
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider


class CrashPluginDelegate : PluginDelegate {

    private var context: Context? = null
    private val crashConfigurationHandler: ConfigurationsHandler by lazy {
        CrashesServiceLocator.getCrashConfigurationHandler()
    }
    private val nonFatalsConfigurationsProvider: NonFatalsConfigurationProvider by lazy {
        CrashesServiceLocator.nonFatalsConfigurationsProvider
    }
    private val nonFatalsReproConfigurationsHandler: NonFatalsReproConfigurationHandler by lazy {
        CrashesServiceLocator.nonFatalsReproConfigurationsHandler
    }

    override fun init(context: Context) {
        this.context = context
        crashConfigurationHandler.migrateCurrentConfiguration()
        clearOldCrashesIfNeeded(context)
        setExceptionHandler()
        CommonsLocator.reproProxy.evaluate(nonFatalsConfigurationsProvider)

    }

    override fun start(context: Context) {
        InstabugUncaughtExceptionHandler.setEarlyCapturingMode(false)
        PoolProvider.postIOTaskWithCheck {
            SettingsManager.getInstance()
                ?.reproConfigurations
                ?.let {
                    handleReproStateConfigurations(it.modesMap)
                }
            CrashSettings.init()
            checkEncryptorVersion()
            startUploaderServices()
        }
    }

    override fun wake() {
        //No implementation needed yet
    }

    override fun sleep() {
        //No implementation needed yet
    }

    override fun stop() {
        context = null
        CrashSettings.release()
    }

    @WorkerThread
    private fun startUploaderServices() {
        if (isCrashReportingEnabled() && CrashReportsDbHelper.getCrashesCount() > 0)
            InstabugCrashesUploaderJob.getInstance().start()
    }


    private fun handleFeaturesFetched(featuresResponse: String) {
        crashConfigurationHandler.handleConfiguration(featuresResponse)
        CommonsLocator.reproProxy.evaluate(CrashesServiceLocator.nonFatalsConfigurationsProvider)
    }

    override fun handleSDKCoreEvent(sdkCoreEvent: IBGSdkCoreEvent) {
        when (sdkCoreEvent) {
            NetworkActivated -> if (isCrashReportingEnabled())
                startCrashesUploaderService()

            User.LoggedOut -> clearUserActivities()
            is FeaturesFetched -> handleFeaturesFetched(sdkCoreEvent.response)
            is IBGSdkCoreEvent.ReproState -> handleReproStateConfigurations(sdkCoreEvent.modesMap)
            else -> {}
        }
    }

    private fun clearOldCrashesIfNeeded(context: Context) {
        if (!InstabugCore.isLastSDKStateEnabled(context)) {
            CrashCleanupUtils.deleteAllCrashesAsync()
        }
    }

    private fun setExceptionHandler() {
        if (InstabugUncaughtExceptionHandler.isRegistered) {
            return
        }
        //Setup Uncaught Exception Handler
        InstabugSDKLogger.d(
            Constants.LOG_TAG,
            "setting Uncaught Exception Handler com.instabug.crash.InstabugUncaughtExceptionHandler"
        )
        Thread.setDefaultUncaughtExceptionHandler(InstabugUncaughtExceptionHandler(Instabug.getApplicationContext()))
    }

    @WorkerThread
    private fun checkEncryptorVersion() {
        InstabugSDKLogger.v(Constants.LOG_TAG, "CrashPlugin checking EncryptorVersion")
        if (CrashSettings.getInstance().isFirstRunAfterEncryptorUpdate) {
            InstabugSDKLogger.v(
                Constants.LOG_TAG,
                "CrashPlugin checking EncryptorVersion > firstRunAfterEncryptorUpdate"
            )
            /*
            * Due to the latest changes in Encryption/Decryption processing, previously the process
            * was done for the whole file vs now the Process is done
            * for the first 256 bytes only no matter how big the file is, it was decided to drop Crashes BD table.
           */
            CrashReportsDbHelper.deleteAll()
            CrashSettings.getInstance().setFirstRunAfterEncryptorUpdate(false)
        }
    }

    private fun clearUserActivities() {
        if (PersistableSettings.getInstance() == null) return
        PersistableSettings.getInstance().lastCrashTime = 0L
    }

    private fun startCrashesUploaderService() {
        if (context != null) {
            PoolProvider.postIOTask {
                CrashCleanupUtils.cleanStaleStateFiles()
                val crashesCount = CrashReportsDbHelper.getCrashesCount()
                if (crashesCount > 0) {
                    CrashReportsDbHelper.trim()
                    InstabugCrashesUploaderJob.getInstance().start()
                }
            }
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Context is null.")
        }
    }

    private fun handleReproStateConfigurations(modesMap: Map<Int, Int>) {
        nonFatalsReproConfigurationsHandler.handle(modesMap)
        CommonsLocator.reproProxy.evaluate(nonFatalsConfigurationsProvider)
    }
}