package com.instabug.commons.caching

import android.content.Context
import androidx.annotation.Keep
import androidx.annotation.WorkerThread
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.snapshot.ifNotExists
import com.instabug.library.util.threading.OrderedExecutorService
import java.io.File

/**
 * A generic contract for providing caching directories in an organized manner
 */
@Keep
interface FileCacheDirectory {
    /**
     * The current saving directory as [File].
     * Could be null if there's no current caching directory.
     */
    val fileDirectory: File?

    /**
     * Deletes the directory referenced by [fileDirectory] with all its contents.
     */
    @WorkerThread
    fun deleteFileDir() {
        fileDirectory?.takeIf { it.exists() }?.deleteRecursively()
    }
}

/**
 * An extension to {FileCacheDirectory} contract for session-based directories
 */
@Keep
interface SessionCacheDirectory : FileCacheDirectory {
    /**
     * The current session directory as [File].
     * Could be null if there's no running session.
     */
    val currentSessionDirectory: File?

    /**
     * The old sessions directories saved in the parent directory excluding the currently running
     * session directory.
     */
    val oldSessionsDirectories: List<File>

    /**
     * Sets the current sessionId for use in creating a directory.
     */
    fun setCurrentSessionId(sessionId: String?)

    /**
     * Adds a watcher by ID to block cleansing if not consented.
     * @param watcherId the watcher id to be added.
     */
    fun addWatcher(watcherId: Int)

    /**
     * Consent on cleansing by a given watcher ID.
     * @param watcherId the watcher id to consider the consent from.
     */
    fun consentOnCleansing(watcherId: Int)

    /**
     * Removes a watcher from the watchers list, hence, stop blocking for its consent.
     * @param watcherId the watcher ID to be removed.
     */
    fun removeWatcher(watcherId: Int)

    /**
     * Query the state of a given watcher by its id.
     * @return a nullable [Boolean] value. If null, means the watcher is not added
     */
    fun queryWatcherConsent(watcherId: Int): Boolean?
}

private const val FILE_CACHING_QUEUE_ID = "crashes-file-caching-exec"

class CrashesCacheDir(
    private val executor: OrderedExecutorService,
    private val ctxGetter: () -> Context?,
    private val attachmentsInternalDirGetter: (Context?) -> File?,
    private val attachmentsExternalDirGetter: (Context?) -> File?
) : SessionCacheDirectory {

    private val attachmentsInternalDir: File? by lazy { attachmentsInternalDirGetter(ctxGetter()) }

    private val attachmentsExternalDir: File? by lazy { attachmentsExternalDirGetter(ctxGetter()) }

    private var currentSessionId: String? = null

    private val watchersMap: MutableMap<Int, Boolean> = mutableMapOf()

    override val fileDirectory: File?
        @WorkerThread get() = attachmentsInternalDir?.let(::getCrashesDirectory)

    private val fileExternalDirectory: File?
        @WorkerThread get() = attachmentsExternalDir?.let(::getCrashesDirectory)

    override val currentSessionDirectory: File?
        @WorkerThread get() = executor.submit(FILE_CACHING_QUEUE_ID) {
            currentSessionId?.let { sessionId ->
                attachmentsInternalDir?.let { base -> getSessionDirectory(base, sessionId) }
            }
        }.get()

    override val oldSessionsDirectories: List<File>
        @WorkerThread get() = executor.submit(FILE_CACHING_QUEUE_ID) { getOldDirs() }.get()

    override fun setCurrentSessionId(sessionId: String?): Unit =
        executor.execute(FILE_CACHING_QUEUE_ID) {
            sessionId.also { currentSessionId = sessionId }
                .also { cleanseIfNeeded() }
                ?.also { markSessionStarter(it) }
        }

    override fun addWatcher(watcherId: Int) = executor.execute(FILE_CACHING_QUEUE_ID) {
        if (!watchersMap.containsKey(watcherId)) {
            watchersMap[watcherId] = false
            "Watcher $watcherId added to crashes dir".logVerbose()
        }
    }

    override fun consentOnCleansing(watcherId: Int) = executor.execute(FILE_CACHING_QUEUE_ID) {
        watchersMap[watcherId] = true
        "Considered consent of id -> $watcherId".logVerbose()
        cleanseIfNeeded()
    }

    override fun removeWatcher(watcherId: Int) = executor.execute(FILE_CACHING_QUEUE_ID) {
        watchersMap.remove(watcherId)
        "Watcher $watcherId removed from crashes dir".logVerbose()
        cleanseIfNeeded()
    }

    override fun deleteFileDir() {
        executor.execute(FILE_CACHING_QUEUE_ID) {
            super.deleteFileDir() //Delete file internal directory
            fileExternalDirectory?.takeIf { it.exists() }?.deleteRecursively()
        }
    }

    override fun queryWatcherConsent(watcherId: Int): Boolean? = watchersMap[watcherId]

    private fun cleanseIfNeeded() {
        runCatching {
            if (watchersMap.any { (_, hasConsented) -> !hasConsented }) return
            "Cleansing crashes directory excluding $currentSessionId".logVerbose()
            fileDirectory?.listFiles { file ->
                file.name != currentSessionId
            }?.forEach { it.deleteRecursively() }
            fileExternalDirectory?.listFiles { file ->
                file.name != currentSessionId
            }?.forEach { it.deleteRecursively() }
            watchersMap.keys.forEach { watchersMap[it] = false }
        }
    }

    private fun getOldDirs(): List<File> = runCatching {
        fileDirectory.oldDirsFiles() + fileExternalDirectory.oldDirsFiles()
    }.getOrElse {
        emptyList()
    }

    private fun File?.oldDirsFiles() = takeIf { it != null && it.exists() }
        ?.listFiles { file -> file.name != currentSessionId }
        ?.toMutableList()
        ?.run(::trimIfNeeded)
        ?: emptyList()

    private fun trimIfNeeded(oldDirectories: List<File>): List<File> {
        val filteredDirs = oldDirectories.asSequence()
            .map { (it to getSessionStarterFile(it)) }
            .onEach { (dir, starterFile) -> if (starterFile == null) dir.deleteRecursively() }
            .filter { (_, starterFile) -> starterFile == null }
            .sortedByDescending { (_, starterFile) ->
                starterFile?.name?.removeSuffix(CRASHES_SESSION_START_MARKER)?.toLong()
            }.map { (dir, _) -> dir }.toMutableList()
        if (filteredDirs.size <= MAX_SESSION_DIRS) return oldDirectories
        val delta = filteredDirs.size - MAX_SESSION_DIRS
        repeat(delta) { filteredDirs.removeLastOrNull()?.deleteRecursively() }
        return filteredDirs
    }

    private fun markSessionStarter(sessionId: String) {
        attachmentsInternalDir?.let { base -> getSessionDirectory(base, sessionId) }
            ?.ifNotExists { mkdirs() }
            ?.let { sessionDir -> getSessionStarterFile(sessionDir, System.currentTimeMillis()) }
            ?.createNewFile()
    }

    companion object {
        const val CRASHES_DIR_NAME = "crashes"
        const val CRASHES_SESSION_START_MARKER = "-sst"
        private const val MAX_SESSION_DIRS = 100
        fun getCrashesDirectory(baseDirectory: File): File =
            baseDirectory.run { File("${absolutePath}${File.separator}$CRASHES_DIR_NAME") }

        fun getSessionDirectory(baseDirectory: File, sessionId: String): File =
            getCrashesDirectory(baseDirectory).run { File("${absolutePath}${File.separator}$sessionId") }

        fun getSessionStarterFile(sessionDir: File, startTime: Long) =
            sessionDir.run { File("$absolutePath${File.separator}$startTime$CRASHES_SESSION_START_MARKER") }

        fun getSessionStarterFile(sessionDir: File): File? =
            sessionDir.listFiles { file ->
                file.name.endsWith(CRASHES_SESSION_START_MARKER)
            }?.firstOrNull()
    }
}
