package com.instabug.terminations.di

import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.commons.IncidentDetectorsListenersRegistry
import com.instabug.commons.caching.FileCacheDirectory
import com.instabug.commons.caching.SessionCacheDirectory
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.diagnostics.di.DiagnosticsLocator
import com.instabug.commons.diagnostics.reporter.DiagnosticsReporter
import com.instabug.commons.session.SessionLinker
import com.instabug.commons.snapshot.CaptorsRegistry
import com.instabug.commons.utils.isAtLeastRunningR
import com.instabug.crash.settings.CrashSettings
import com.instabug.library.InstabugNetworkJob
import com.instabug.library.WatchableSpansCacheDirectory
import com.instabug.library.core.eventbus.CurrentActivityLifeCycleEventBus
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.logscollection.DataWatcher
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.networkv2.limitation.RateLimiter
import com.instabug.library.tracking.FirstFGTimeProvider
import com.instabug.library.util.threading.PoolProvider
import com.instabug.terminations.*
import com.instabug.terminations.cache.TerminationsCacheDir
import com.instabug.terminations.cache.TerminationsCachingManager
import com.instabug.terminations.cache.TerminationsCachingManagerImpl
import com.instabug.terminations.configuration.TerminationsConfigurationHandler
import com.instabug.terminations.configuration.TerminationsConfigurationProvider
import com.instabug.terminations.configuration.TerminationsConfigurationProviderImpl
import com.instabug.terminations.model.Termination
import com.instabug.terminations.sync.TerminationsSyncJob
import java.lang.ref.WeakReference
import java.util.concurrent.Executor
import java.util.concurrent.ScheduledExecutorService

object ServiceLocator {
    private val objectsMap = mutableMapOf<String, WeakReference<Any?>?>()

    val settingsManager: CrashSettings
        get() = CrashSettings.getInstance()

    val cachingManager: TerminationsCachingManager
        @Synchronized get() = TerminationsCachingManager::class.toString().let { key ->
            getIfAvailable(key)?.let { existing -> existing as TerminationsCachingManager }
                ?: run { TerminationsCachingManagerImpl().also { impl -> save(key, impl) } }
        }

    val syncJob: InstabugNetworkJob
        @Synchronized get() = InstabugNetworkJob::class.toString().let { key ->
            getIfAvailable(key)?.let { existing -> existing as InstabugNetworkJob }
                ?: run { TerminationsSyncJob().also { impl -> save(key, impl) } }
        }

    val networkManager: NetworkManager
        @Synchronized get() = NetworkManager::class.toString().let { key ->
            getIfAvailable(key)?.let { existing -> existing as NetworkManager }
                ?: run { NetworkManager().also { impl -> save(key, impl) } }
        }

    @Synchronized
    fun getRateLimiter(onLimited: (Termination) -> Unit): RateLimiter<Termination, CrashSettings> =
        RateLimiter::class.toString().let { key ->
            getIfAvailable(key)?.let { existing -> existing as RateLimiter<Termination, CrashSettings> }
                ?: run { RateLimiter(settingsManager, onLimited).also { impl -> save(key, impl) } }
        }

    private fun getIfAvailable(key: String): Any? = objectsMap[key]?.get()

    private fun save(key: String, value: Any) =
        run { objectsMap[key] = WeakReference(value) }

    val operationsExecutor: Executor
        get() = PoolProvider.getSingleThreadExecutor("termination-operations-executor")

    val terminationStoreLimit: Int
        get() = Constants.TERMINATIONS_STORE_LIMIT

    val terminationsConfigurationProvider: TerminationsConfigurationProvider
            by lazy { TerminationsConfigurationProviderImpl() }

    val terminationsConfigurationHandler: ConfigurationsHandler
            by lazy { TerminationsConfigurationHandler() }

    val crashesCacheDir: SessionCacheDirectory
        get() = CommonsLocator.crashesCacheDir

    val terminationsCacheDir: FileCacheDirectory by lazy { TerminationsCacheDir(crashesCacheDir) }

    val captorsRegistry: CaptorsRegistry
        get() = CommonsLocator.captorsRegistry

    private val postAndroidRMigrator: TerminationMigrator
        @RequiresApi(Build.VERSION_CODES.R)
        get() = PostAndroidRMigrator(
            appCtx,
            crashesCacheDir,
            DefaultTerminationsValidator.Factory(terminationsConfigurationProvider),
            firstFGProvider,
            cachingManager,
            reproScreenshotsCacheDir
        )

    private val preAndroidRMigrator: TerminationMigrator
        get() = PreAndroidRMigrator(
            appCtx,
            crashesCacheDir,
            DefaultTerminationsValidator.Factory(terminationsConfigurationProvider),
            firstFGProvider,
            cachingManager,
            reproScreenshotsCacheDir
        )

    val terminationsMigrator: TerminationMigrator
        get() = if (isAtLeastRunningR) postAndroidRMigrator else preAndroidRMigrator

    val detectorsListenersRegistry: IncidentDetectorsListenersRegistry
        get() = CommonsLocator.detectorsListenersRegistry

    val appCtx: Context?
        get() = CommonsLocator.appCtx

    val firstFGProvider: FirstFGTimeProvider
        get() = FirstFGTimeProvider.Factory()

    val currentActivityLifeCycleEventBus: CurrentActivityLifeCycleEventBus
        get() = CurrentActivityLifeCycleEventBus

    val sessionLinker: SessionLinker
        get() = CommonsLocator.sessionLinker

    val diagnosticsReporter: DiagnosticsReporter
        get() = DiagnosticsLocator.reporter

    val reproScreenshotsCacheDir: WatchableSpansCacheDirectory
        get() = CommonsLocator.reproScreenshotsCacheDir

    val hubDataWatcher: DataWatcher
        get() = CoreServiceLocator.hubDataWatcher

    fun getScheduledExecutor(name: String): ScheduledExecutorService =
        CommonsLocator.getScheduledExecutor(name)
}
