package com.instabug.terminations

import android.content.Context
import androidx.annotation.WorkerThread
import com.instabug.commons.PluginDelegate
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.models.Incident.Type
import com.instabug.commons.snapshot.StateSnapshotCaptor
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.NetworkActivated
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.sessionV3.sync.AllFilter
import com.instabug.library.sessionV3.sync.NoneFilter
import com.instabug.library.settings.SettingsManager
import com.instabug.library.tracking.ActivityLifeCycleEvent
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.visualusersteps.ReproCapturingProxy
import com.instabug.terminations.Constants.SNAPSHOT_SYSTEM_ID
import com.instabug.terminations.di.ServiceLocator
import com.instabug.terminations.di.ServiceLocator.captorsRegistry
import com.instabug.terminations.di.ServiceLocator.crashesCacheDir
import com.instabug.terminations.di.ServiceLocator.currentActivityLifeCycleEventBus
import com.instabug.terminations.di.ServiceLocator.diagnosticsReporter
import com.instabug.terminations.di.ServiceLocator.hubDataWatcher
import com.instabug.terminations.di.ServiceLocator.operationsExecutor
import com.instabug.terminations.di.ServiceLocator.reproScreenshotsCacheDir
import com.instabug.terminations.di.ServiceLocator.sessionLinker
import com.instabug.terminations.di.ServiceLocator.syncJob
import com.instabug.terminations.di.ServiceLocator.terminationsCacheDir
import com.instabug.terminations.di.ServiceLocator.terminationsConfigurationHandler
import com.instabug.terminations.di.ServiceLocator.terminationsConfigurationProvider
import com.instabug.terminations.diagnostics.TerminationIncidentType

class TerminationsPluginDelegate : PluginDelegate {

    private var lifecycleDisposable: IBGDisposable? = null
    private var isLastEnabled: Boolean = false

    private val reproProxy: ReproCapturingProxy by lazy {
        CommonsLocator.reproProxy
    }


    override fun init(context: Context) {
        addMutualDirsWatcher()
        terminationsConfigurationHandler.migrateCurrentConfiguration()
        reproProxy.evaluate(terminationsConfigurationProvider)

    }

    override fun start(context: Context) {
        PoolProvider.postIOTaskWithCheck {
            SettingsManager.getInstance()
                ?.reproConfigurations
                ?.let {
                    handleReproStateConfigurations(it.modesMap)
                }
        }
        isLastEnabled = terminationsConfigurationProvider.isEnabled()
        if (!isLastEnabled) removeMutualDirsWatcher()
    }

    override fun wake() {
        if (!isLastEnabled) return
        operateOnExec {
            startSnapshotCaptorsIfPossible()
            createSessionWeakLink()
            if (migrateAndSync() is MigrationResult.Failed) {
                "Terminations migration failed on wake, subscribing to lifecycle".logVerbose()
                subscribeToActivityLifecycleEvents()
            }
        }
    }

    override fun sleep() {
        operateOnExec { internalStop(stopTrmCaptor = false) }
    }

    override fun stop() {
        operateOnExec {
            internalStop(stopTrmCaptor = true)
            consentOnMutualDirsCleansing()
        }
    }

    private fun handleFeaturesFetchedEvent() {
        "Terminations received features".logVerbose()
        operateOnExec(this::handleTerminationStateChanged)
    }

    override fun handleSDKCoreEvent(sdkCoreEvent: IBGSdkCoreEvent) {
        when (sdkCoreEvent) {
            is FeaturesFetched -> handleFeaturesFetched(sdkCoreEvent.response)
            NetworkActivated -> handleNetworkActivated()
            is Features -> handleFeaturesFetchedEvent()
            is IBGSdkCoreEvent.ReproState -> handleReproStateConfigurations(sdkCoreEvent.modesMap)
            else -> {}
        }
    }

    private fun handleNetworkActivated() {
        "Terminations received network activated".logVerbose()
        startSyncingIfPossible()
    }

    private fun handleFeaturesFetched(featuresResponse: String) {
        "Terminations received features fetched".logVerbose()
        operateOnExec {
            reproProxy.evaluate(terminationsConfigurationProvider)
            terminationsConfigurationHandler.handleConfiguration(featuresResponse)
            handleTerminationStateChanged()
        }
    }

    private fun operateOnExec(operation: () -> Unit) {
        operationsExecutor.execute(operation)
    }

    private fun subscribeToActivityLifecycleEvents() {
        lifecycleDisposable =
            currentActivityLifeCycleEventBus.subscribe(::onActivityEvent)
    }

    private fun onActivityEvent(event: ActivityLifeCycleEvent) {
        "Terminations received lifecycle event $event".logVerbose()
        if (event != ActivityLifeCycleEvent.STARTED) return
        operateOnExec(this::migrateAndSync)
        lifecycleDisposable?.dispose()
        lifecycleDisposable = null
    }

    @WorkerThread
    private fun handleTerminationStateChanged() {
        if (terminationsConfigurationProvider.isEnabled() == isLastEnabled) return
        if (terminationsConfigurationProvider.isEnabled()) {
            isLastEnabled = true
            "Terminations is enabled".logVerbose()
            createSessionWeakLink()
            startSnapshotCaptorsIfPossible()
            addMutualDirsWatcher()
            migrateAndSync()
            return
        }
        isLastEnabled = false
        "Terminations is disabled, clearing..".logVerbose()
        validateCurrentSessionWeakLink()
        internalStop(stopTrmCaptor = true)
        terminationsCacheDir.deleteFileDir()
        with(ServiceLocator) {
            appCtx?.let { ctx -> cachingManager.clear(ctx) }
            removeMutualDirsWatcher()
        }
    }

    @WorkerThread
    private fun internalStop(stopTrmCaptor: Boolean) {
        stopSnapshotCaptors(stopTrmCaptor)
        lifecycleDisposable?.dispose()
        lifecycleDisposable = null
    }

    @WorkerThread
    private fun startSnapshotCaptorsIfPossible() {
        if (!isLastEnabled) return
        with(captorsRegistry) {
            start(SNAPSHOT_SYSTEM_ID, StateSnapshotCaptor.Factory())
            start(SNAPSHOT_SYSTEM_ID, AbstractTerminationSnapshotCaptor.Factory())
        }
    }

    @WorkerThread
    private fun stopSnapshotCaptors(stopTrmCaptor: Boolean) {
        with(captorsRegistry) {
            stop(SNAPSHOT_SYSTEM_ID, StateSnapshotCaptor.ID)
            if (!stopTrmCaptor) return
            stop(SNAPSHOT_SYSTEM_ID, AbstractTerminationSnapshotCaptor.ID)
        }
    }

    @WorkerThread
    private fun migrateAndSync(): MigrationResult = with(ServiceLocator) {
        terminationsMigrator()
            .also { "Trm migration result $it".logVerbose() }
            .takeIf { result -> result is MigrationResult.Migrated }
            ?.also { consentOnMutualDirsCleansing() }
            ?.also(this@TerminationsPluginDelegate::reportedDiagnosticsCapture)
            ?.also(this@TerminationsPluginDelegate::validateWeakLinks)
            ?.also(this@TerminationsPluginDelegate::notifyDataReadiness)
            ?.also { startSyncingIfPossible() } ?: MigrationResult.Failed
    }

    @WorkerThread
    private fun createSessionWeakLink() {
        InstabugCore.getRunningSession()
            ?.let { session -> sessionLinker.weakLink(session.id, Type.Termination) }
    }

    @WorkerThread
    private fun validateCurrentSessionWeakLink() {
        InstabugCore.getRunningSession()
            ?.let { session -> sessionLinker.validateWeakLink(session.id, null, Type.Termination) }
    }

    @WorkerThread
    private fun validateWeakLinks(result: MigrationResult) {
        if (result !is MigrationResult.Migrated) return
        result.incidents.forEach { trm ->
            sessionLinker.validateWeakLink(trm.sessionId, trm.metadata.uuid, trm.type)
        }
        result.migratedSessions.forEach {
            sessionLinker.validateWeakLink(it, null, Type.Termination)
        }
    }

    @WorkerThread
    private fun notifyDataReadiness(result: MigrationResult) {
        if (result !is MigrationResult.Migrated) return
        val batchingFilter =
            result.incidents.size.takeIf { it > 0 }?.let { NoneFilter } ?: AllFilter
        InstabugCore.notifyV3SessionDataReadiness(batchingFilter)
    }

    @WorkerThread
    private fun reportedDiagnosticsCapture(result: MigrationResult) {
        if (result !is MigrationResult.Migrated) return
        result.incidents.map {
            CalibrationDiagnosticEvent(
                TerminationIncidentType(),
                CalibrationDiagnosticEvent.Action.Captured
            )
        }.forEach(diagnosticsReporter::report)
    }

    private fun startSyncingIfPossible() {
        if (!isLastEnabled) return
        syncJob.start()
    }

    private fun addMutualDirsWatcher() {
        crashesCacheDir.addWatcher(SNAPSHOT_SYSTEM_ID)
        reproScreenshotsCacheDir.addWatcher(SNAPSHOT_SYSTEM_ID)
        hubDataWatcher.addWatcher(SNAPSHOT_SYSTEM_ID)
    }

    private fun removeMutualDirsWatcher() {
        crashesCacheDir.removeWatcher(SNAPSHOT_SYSTEM_ID)
        reproScreenshotsCacheDir.removeWatcher(SNAPSHOT_SYSTEM_ID)
        hubDataWatcher.removeWatcher(SNAPSHOT_SYSTEM_ID)
    }

    private fun consentOnMutualDirsCleansing() {
        crashesCacheDir.consentOnCleansing(SNAPSHOT_SYSTEM_ID)
        reproScreenshotsCacheDir.consentOnCleansing(SNAPSHOT_SYSTEM_ID)
        hubDataWatcher.consentOnCleansing(SNAPSHOT_SYSTEM_ID)
    }

    private fun handleReproStateConfigurations(modesMap: Map<Int, Int>) {
        terminationsConfigurationHandler.handle(modesMap)
        reproProxy.evaluate(terminationsConfigurationProvider)
    }
}
