package com.instabug.fatalhangs.sync

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.instabug.crash.Constants
import com.instabug.crash.settings.CrashSettings
import com.instabug.crash.utils.deleteAttachment
import com.instabug.fatalhangs.di.FatalHangsServiceLocator
import com.instabug.fatalhangs.model.FatalHang
import com.instabug.fatalhangs.model.FatalHangState
import com.instabug.fatalhangs.sync.FatalHangsRequestsBuilder.buildFatalHangLogsRequest
import com.instabug.fatalhangs.sync.FatalHangsRequestsBuilder.buildFatalHangRequest
import com.instabug.fatalhangs.sync.FatalHangsRequestsBuilder.buildSingleAttachmentRequest
import com.instabug.library.IBGNetworkWorker
import com.instabug.library.internal.storage.AttachmentsUtility
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation
import com.instabug.library.model.Attachment
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.networkv2.RateLimitedException
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONException
import org.json.JSONObject

class FatalHangsSyncManagerImpl : FatalHangsSyncManager {

    private val networkManager: NetworkManager by lazy { FatalHangsServiceLocator.getNetworkManager() }
    private val fatalHangsCacheManager = FatalHangsServiceLocator.getFatalHangsCacheManager()

    override fun syncFatalHangs() {
        PoolProvider.getNetworkingSingleThreadExecutor(IBGNetworkWorker.CRASH)
            .execute {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Starting Fatal hangs sync")
                startFatalHangsSync()
            }
    }


    private fun startFatalHangsSync() {
        val context = FatalHangsServiceLocator.getContext()
        context?.let { _context ->
            var fatalHang = fatalHangsCacheManager.retrieveFirst(_context)
            fatalHang?.let {
                when (it.fatalHangState) {
                    FatalHangState.READY_TO_BE_SENT -> {
                        sendFatalHangRequest(it, object : Request.Callbacks<String, Throwable> {
                            override fun onSucceeded(id: String) {
                                CrashSettings.getInstance().setLastRequestStartedAt(0)
                                it.apply {
                                    temporaryServerToken = id
                                    fatalHangState = FatalHangState.LOGS_READY_TO_BE_UPLOADED
                                }
                                fatalHangsCacheManager.update(it)
                                sendFatalHangLogs(it)
                            }

                            override fun onFailed(error: Throwable) {
                                if (error is RateLimitedException)
                                    handleRateLimitException(it, error)
                                else
                                    InstabugSDKLogger.e(
                                        Constants.LOG_TAG,
                                        "Failed to send fatal hang",
                                        error
                                    )
                                fatalHang = null
                            }

                        })
                    }
                    FatalHangState.LOGS_READY_TO_BE_UPLOADED -> {
                        sendFatalHangLogs(it)
                    }
                    FatalHangState.ATTACHMENTS_READY_TO_BE_UPLOADED -> {
                        sendFatalHangAttachments(it)
                    }
                }
            }
        }
    }

    @VisibleForTesting
    fun sendFatalHangRequest(
        fatalHang: FatalHang,
        callback: Request.Callbacks<String, Throwable>
    ) {
        if (CrashSettings.getInstance().isRateLimited) {
            handleRateIsLimited(fatalHang)
            return
        }
        CrashSettings.getInstance().setLastRequestStartedAt(System.currentTimeMillis())
        val request = buildFatalHangRequest(fatalHang)
        networkManager.doRequestOnSameThread(
            RequestType.NORMAL,
            request,
            object : Request.Callbacks<RequestResponse, Throwable> {
                override fun onSucceeded(response: RequestResponse?) {
                    response?.let {
                        InstabugSDKLogger.v(
                            Constants.LOG_TAG, "sendFatalHangRequest Succeeded, Response code: " +
                                    it.responseCode +
                                    "Response body: " + it.responseBody
                        )
                        try {
                            if (it.responseBody != null) {
                                val responseBody = JSONObject(it.responseBody as String)
                                callback.onSucceeded(responseBody.getString("id"))
                            } else {
                                callback.onFailed(
                                    JSONException("response.getResponseBody() returned null")
                                )
                            }
                        } catch (e: JSONException) {
                            InstabugSDKLogger.e(
                                Constants.LOG_TAG,
                                "Couldn't parse Fatal Hang request response.",
                                e
                            )
                        }
                    }
                }

                override fun onFailed(error: Throwable?) {
                    callback.onFailed(error)
                }

            })
    }

    private fun sendFatalHangLogs(fatalHang: FatalHang) {
        val request = buildFatalHangLogsRequest(fatalHang)
        networkManager.doRequestOnSameThread(
            RequestType.NORMAL,
            request,
            object : Request.Callbacks<RequestResponse, Throwable> {
                override fun onSucceeded(response: RequestResponse?) {
                    fatalHang.fatalHangState = FatalHangState.ATTACHMENTS_READY_TO_BE_UPLOADED
                    fatalHangsCacheManager.update(fatalHang)
                    sendFatalHangAttachments(fatalHang)
                }

                override fun onFailed(error: Throwable) {
                    InstabugSDKLogger.e(
                        Constants.LOG_TAG,
                        "Failed to send Fatal hang logs request",
                        error
                    )
                }

            })
    }

    private fun sendFatalHangAttachments(fatalHang: FatalHang) {
        uploadFatalHangAttachments(fatalHang, object : Request.Callbacks<Boolean, Throwable> {
            override fun onSucceeded(response: Boolean?) {
                InstabugSDKLogger.d(
                    Constants.LOG_TAG, "Fatal hang attachments uploaded" +
                            " successfully"
                )

                val context = FatalHangsServiceLocator.getContext()
                context?.let { deleteFatalHang(context, fatalHang) }
            }

            override fun onFailed(error: Throwable) {
                AttachmentsUtility.encryptAttachmentsAndUpdateDb(fatalHang.attachments)
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Something went wrong while uploading fatal hang attachments",
                    error
                )
            }

        })
    }

    private fun uploadFatalHangAttachments(
        fatalHang: FatalHang,
        fatalHangAttachmentsCallbacks: Request.Callbacks<Boolean, Throwable>
    ) {
        InstabugSDKLogger.d(
            Constants.LOG_TAG,
            "Uploading Fatal hang attachments, size: " + fatalHang.attachments.size
        )
        if (fatalHang.attachments.size == 0) {
            fatalHangAttachmentsCallbacks.onSucceeded(true)
            return
        }
        val synced: MutableList<Attachment> = ArrayList()
        for (i in fatalHang.attachments.indices) {
            // create attachment request.
            val attachment = fatalHang.attachments[i]
            val isAttachmentDecrypted = AttachmentsUtility.decryptAttachmentAndUpdateDb(attachment)
            if (isAttachmentDecrypted) {
                val attachmentRequest: Request? =
                    buildSingleAttachmentRequest(fatalHang, attachment)
                attachmentRequest?.let { request ->
                    attachment.localPath?.let { it ->
                        val file = FatalHangsServiceLocator.getFileFromPath(it)
                        if (file.exists() && file.length() > 0) {
                            attachment.attachmentState = Attachment.AttachmentState.SYNCED
                            networkManager.doRequestOnSameThread(
                                RequestType.MULTI_PART,
                                request,
                                object : Request.Callbacks<RequestResponse, Throwable> {
                                    override fun onSucceeded(requestResponse: RequestResponse) {
                                        InstabugSDKLogger.v(
                                            Constants.LOG_TAG,
                                            "uploadingFatalHangAttachmentRequest Succeeded, Response code:" +
                                                    " " + requestResponse.responseCode +
                                                    ", Response body: " + requestResponse.responseBody
                                        )
                                        attachment.localPath?.let {
                                            deleteAttachment(attachment, fatalHang.id)
                                            synced.add(attachment)
                                        }
                                        if (synced.size == fatalHang.attachments.size) {
                                            fatalHangAttachmentsCallbacks.onSucceeded(true)
                                        }
                                    }

                                    override fun onFailed(error: Throwable) {
                                        InstabugSDKLogger.d(
                                            Constants.LOG_TAG,
                                            "uploadingFatalHangAttachmentRequest got error: " + error
                                                .message
                                        )
                                        fatalHangAttachmentsCallbacks.onFailed(error)
                                    }
                                })
                        } else {
                            InstabugSDKLogger.w(
                                Constants.LOG_TAG, "Skipping attachment file of type "
                                        + attachment.type + " because it's either not found or empty file"
                            )
                        }
                    }
                }
            } else {
                InstabugSDKLogger.w(
                    Constants.LOG_TAG, "Skipping attachment file of type "
                            + attachment.type + " because it was not decrypted successfully"
                )
            }
        }
    }

    private fun handleRateIsLimited(fatalHang: FatalHang) {
        logRateIsLimited()
        FatalHangsServiceLocator.getContext()
            ?.let { context -> deleteFatalHang(context, fatalHang) }
    }

    private fun deleteFatalHang(context: Context, fatalHang: FatalHang) {
        runCatching {
            fatalHang.attachments
                .forEach { deleteAttachment(it, fatalHang.id) }
                .also { deleteFataHangAndStateFile(context, fatalHang) }
            fatalHang.getSavingDirOnDisk(context).takeIf { it.exists() }?.deleteRecursively()
        }.onFailure {
            InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't delete fatal hang ${fatalHang.id}", it)
        }
    }

    private fun deleteFataHangAndStateFile(context: Context, fatalHang: FatalHang) {
        if (fatalHang.stateUri != null) {
            InstabugSDKLogger.d(
                Constants.LOG_TAG,
                "attempting to delete state file for Fatal hang with id: " + fatalHang.id
            )
            DiskUtils.with(context)
                .deleteOperation(DeleteUriDiskOperation(fatalHang.stateUri))
                .runCatching { execute() }
                .getOrElse { throwable ->
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Unable to delete state file", throwable)
                    null
                }?.let { status ->
                    InstabugSDKLogger.d(Constants.LOG_TAG, "result:$status")
                    //deleting FatalHang from db
                    InstabugSDKLogger.d(Constants.LOG_TAG, "deleting FatalHang:" + fatalHang.id)
                    fatalHangsCacheManager.delete(fatalHang.id!!)
                    startFatalHangsSync()
                }
        } else {
            InstabugSDKLogger.i(Constants.LOG_TAG, "No state file found. deleting Fatal hang")
            //deleting FatalHang from db
            fatalHangsCacheManager.delete(fatalHang.id!!)
            startFatalHangsSync()
        }

    }

    private fun logRateIsLimited() {
        InstabugSDKLogger.d(
            Constants.LOG_TAG,
            RateLimitedException.RATE_LIMIT_REACHED.format(Constants.FEATURE_NAME)
        )
    }

    private fun handleRateLimitException(fatalHang: FatalHang, exception: RateLimitedException) {
        CrashSettings.getInstance().setLimitedUntil(exception.period)
        handleRateIsLimited(fatalHang)
    }
}
