package com.instabug.commons.snapshot

import android.content.Context
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.lifecycle.CompositeLifecycleOwner
import com.instabug.commons.lifecycle.CompositeLifecycleOwner.CompositeLifecycleObserver
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.utils.updateScreenShotAnalytics
import com.instabug.commons.utils.generateReproConfigsMap
import com.instabug.library.model.State
import com.instabug.library.util.threading.defensiveLog
import com.instabug.library.util.threading.reportOOM
import com.instabug.library.visualusersteps.ReproConfigurationsProvider
import java.io.File
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.TimeUnit

/**
 * Generic contract for captors.
 * A captor represents a periodic capturing mechanism for files, database updates, etc ...
 */
interface Captor {
    /**
     * Returns the id of the captor. Should be unique and saved for further use by the [CaptorsRegistry].
     */
    val id: Int

    /**
     * Indicates whether the captor has been shutdown or not.
     */
    val isShutdown: Boolean

    /**
     * Starts the captor with either a preconfigured period or period configured upon initialization.
     */
    fun start()

    /**
     * Shuts down the captor for good. Shutting down the captor prevents further start commands.
     */
    fun shutdown()

    /**
     * Forces the captor to execute its preconfigured job immediately.
     */
    fun force()
}

abstract class AbstractCaptor(
    private val executorFactory: (String) -> ScheduledExecutorService
) : Captor {
    protected abstract val captorName: String
    protected abstract val capturingPeriod: Long

    protected val scheduledExecutor: ScheduledExecutorService
            by lazy { executorFactory("$captorName$EXECUTOR_NAME_SUFFIX") }

    private var scheduledJob: ScheduledFuture<*>? = null

    final override val isShutdown: Boolean
        get() = scheduledExecutor.isShutdown

    private val isRunning: Boolean
        get() = !(scheduledJob?.isCancelled ?: true)

    protected abstract fun capture()

    protected open fun onStart() {
        //To be implemented for on start operations
    }

    protected open fun onShutdown() {
        //To be implemented for on shut operations
    }

    final override fun start() = synchronized(this) {
        if (!internalStart(0)) return
        onStart()
    }

    final override fun shutdown(): Unit = synchronized(this) {
        if (isShutdown) return
        runCatching(this::onShutdown)
        runCatching {
            internalPause()
            scheduledExecutor.shutdownNow()
        }
    }

    final override fun force(): Unit = synchronized(this) {
        if (isShutdown) return
        runCatching {
            internalPause()
            scheduledExecutor.execute { capture(); internalStart(capturingPeriod) }
        }
    }

    protected fun internalStart(withDelay: Long): Boolean {
        if (isRunning || isShutdown) return false
        scheduledJob = scheduledExecutor.scheduleCaptor(this::captureSafely, withDelay)
        return true
    }

    private fun captureSafely() {
        runCatching( this::capture).onFailure {error ->
            if (error is InterruptedException) {
                throw error
            } else {
                defensiveLog(error)
                error.takeIf { it is OutOfMemoryError }
                    ?.let { it as OutOfMemoryError }
                    ?.also { reportOOM(it) }
            }
        }
    }
    protected fun internalPause(): Boolean {
        if (!isRunning || isShutdown) return false
        scheduledJob?.cancel(true)
        scheduledJob = null
        return true
    }

    private fun ScheduledExecutorService.scheduleCaptor(captor: Runnable, delay: Long = 0L) =
        scheduleAtFixedRate(captor, delay, capturingPeriod, TimeUnit.SECONDS)

    companion object {
        private const val EXECUTOR_NAME_SUFFIX = "CaptorExecutor"
    }
}

class CaptorConfigurations(
    private val ctxGetter: () -> Context?,
    private val savingDirectoryGetter: () -> File?,
    val executorFactory: (String) -> ScheduledExecutorService
) {
    val ctx: Context?
        get() = ctxGetter()
    val savingDirectory: File?
        get() = savingDirectoryGetter()
}

class StateSnapshotCaptor(
    private val configurations: CaptorConfigurations,
    private val lifecycleOwner: CompositeLifecycleOwner,
) : AbstractCaptor(configurations.executorFactory), CompositeLifecycleObserver {
    override val id: Int
        get() = ID
    override val captorName: String
        get() = CAPTOR_NAME
    override val capturingPeriod: Long
        get() = 5L

    private val File.snapshotFile: File
        get() = File("$absolutePath${File.separator}$STATE_SNAPSHOT_FILE_NAME")

    private val File.oldSnapshotFile: File
        get() = File("$absolutePath$OLD_STATE_SNAPSHOT_FILE_SUFFIX")

    override fun onStart() {
        lifecycleOwner.register(this)
        "Starting state snapshot captor".logVerbose()
    }

    override fun onShutdown() {
        lifecycleOwner.unregister(this)
        "Shutting down state snapshot captor".logVerbose()
    }

    override fun capture() {
        if (Thread.currentThread().isInterrupted) return
        configurations.savingDirectory?.run {
            //Rename old snapshot file to be suffixed with -old
            val oldSnapshotFile = snapshotFile.takeIf { it.exists() }
                ?.run { oldSnapshotFile.also { renameTo(it) } }

            snapshotFile.parentFile?.ifNotExists { mkdirs() }

            //Create new snapshot
            configurations.ctx?.let { ctx ->
                State.Builder(ctx)
                    .build(true, true, true,1.0f, false)
                    .apply(State::updateVisualUserSteps)
                    .apply(State::generateReproConfigsMap)
                    .apply(this@StateSnapshotCaptor::updateScreenShotAnalyticsData)
                    .let(snapshotFile::writeSerializable)
            }

            //Delete old snapshot
            oldSnapshotFile?.delete()
        }
    }

    private fun updateScreenShotAnalyticsData(state: State): State =
        state.apply {
            this.updateScreenShotAnalytics()
        }


    override fun onActivityStarted() {
        "StateSnapshotCaptor: Activity started".logVerbose()
        force()
    }

    override fun onFragmentStarted() {
        "StateSnapshotCaptor: Fragment started".logVerbose()
        force()
    }

    object Factory {
        @JvmStatic
        @JvmOverloads
        operator fun invoke(
            ctxGetter: () -> Context? = CommonsLocator::appCtx,
            savingDirectoryGetter: () -> File? = CommonsLocator.crashesCacheDir::currentSessionDirectory,
            executorFactory: (String) -> ScheduledExecutorService = CommonsLocator::getScheduledExecutor,
            lifecycleOwner: CompositeLifecycleOwner = CommonsLocator.compositeLifecycleOwner,
        ) = StateSnapshotCaptor(
            configurations = CaptorConfigurations(
                ctxGetter,
                savingDirectoryGetter,
                executorFactory
            ),
            lifecycleOwner = lifecycleOwner
        )
    }

    companion object {
        const val ID = 0x001
        const val STATE_SNAPSHOT_FILE_NAME = "snapshot"
        const val OLD_STATE_SNAPSHOT_FILE_SUFFIX = "-old"
        private const val CAPTOR_NAME = "CrashesStateSnapshot"

        fun getSnapshotFile(sessionDirectory: File): File =
            sessionDirectory.run { File("$absolutePath${File.separator}$STATE_SNAPSHOT_FILE_NAME") }

        fun getOldSnapshotFile(sessionDirectory: File): File =
            getSnapshotFile(sessionDirectory).run { File("$absolutePath$OLD_STATE_SNAPSHOT_FILE_SUFFIX") }
    }
}
