package com.instabug.anr.network;

import static com.instabug.anr.model.AnrState.ATTACHMENTS_READY_TO_BE_UPLOADED;
import static com.instabug.anr.model.AnrState.LOGS_READY_TO_BE_UPLOADED;
import static com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent.Action;
import static com.instabug.crash.utils.DeleteCrashUtilsKt.deleteAnrAndStateFile;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_TEMPORARY_SERVER_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_UPLOAD_STATE;

import android.content.ContentValues;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.anr.cache.AnrReportsDbHelper;
import com.instabug.anr.diagnostics.ANRIncidentType;
import com.instabug.anr.model.Anr;
import com.instabug.anr.model.AnrState;
import com.instabug.commons.di.CommonsLocator;
import com.instabug.commons.diagnostics.di.DiagnosticsLocator;
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent;
import com.instabug.crash.Constants;
import com.instabug.crash.OnCrashSentCallback;
import com.instabug.crash.models.CrashMetadata;
import com.instabug.crash.settings.CrashSettings;
import com.instabug.crash.utils.DeleteCrashUtilsKt;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;

import java.util.List;

public class InstabugAnrUploaderJob extends InstabugNetworkJob {

    private static final String TAG = "InstabugAnrUploaderJob";
    @Nullable
    private static InstabugAnrUploaderJob INSTANCE;

    private InstabugAnrUploaderJob() {
    }

    public synchronized static InstabugAnrUploaderJob getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InstabugAnrUploaderJob();
        }

        return INSTANCE;
    }

    private static void uploadAnrs(@NonNull final Context context) throws JSONException {

        List<Anr> anrs = AnrReportsDbHelper.retrieve(context);
        InstabugSDKLogger.v(Constants.LOG_TAG, "Found " + anrs.size() + " ANRs in cache");
        for (final Anr anr : anrs) {
            if (anr.getAnrState() == AnrState.READY_TO_BE_SENT) {
                if (CrashSettings.getInstance().isRateLimited()) {
                    deleteAnr(context, anr);
                    logRateIsLimited();
                    continue;
                }
                CrashSettings.getInstance().setLastRequestStartedAt(System.currentTimeMillis());
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading anr: " + anr.toString());
                AnrsService.getInstance().reportAnr(anr, new Request.Callbacks<String, Throwable>() {
                    @Override
                    public void onSucceeded(@Nullable String temporaryServerToken) {
                        if (temporaryServerToken == null) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "null response, aborting...");
                            return;
                        }
                        CrashSettings.getInstance().setLastRequestStartedAt(0L);

                        InstabugSDKLogger.d(Constants.LOG_TAG, "ANR uploaded successfully");
                        anr.setTemporaryServerToken(temporaryServerToken);
                        anr.setAnrState(LOGS_READY_TO_BE_UPLOADED);
                        onAnrSent(anr);

                        //updating ANR in db
                        ContentValues contentValues = new ContentValues();
                        contentValues.put(COLUMN_ANR_TEMPORARY_SERVER_TOKEN, temporaryServerToken);
                        contentValues.put(COLUMN_ANR_UPLOAD_STATE, LOGS_READY_TO_BE_UPLOADED);
                        AnrReportsDbHelper.update(anr.getId(), contentValues);

                        uploadAnrLogs(anr);
                    }

                    @Override
                    public void onFailed(Throwable error) {
                        if (error instanceof RateLimitedException)
                            handleRateLimitedException((RateLimitedException) error, anr, context);
                        else
                            InstabugSDKLogger.d(Constants.LOG_TAG, "Something went wrong while uploading ANR, " + error.getMessage());
                    }
                });
            } else if (anr.getAnrState() == LOGS_READY_TO_BE_UPLOADED) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "ANR: " + anr.toString() +
                        " already uploaded but has unsent logs, uploading now");
                uploadAnrLogs(anr);
            } else if (anr.getAnrState() == ATTACHMENTS_READY_TO_BE_UPLOADED) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "ANR: " + anr.toString() +
                        " already uploaded but has unsent attachments, uploading now");
                uploadAnrAttachments(anr);
            }
        }
    }

    private static void onAnrSent(Anr anr) {
        OnCrashSentCallback crashMetadataCallback = CommonsLocator.getCrashMetadataCallback();
        CrashMetadata crashMetaData = CommonsLocator.getCrashMetadataMapper().toMetadata(anr);
        crashMetadataCallback.onCrashSent(crashMetaData);
    }

    private static void handleRateLimitedException(RateLimitedException exception, @NonNull Anr anr, Context context) {
        CrashSettings.getInstance().setLimitedUntil(exception.getPeriod());
        logRateIsLimited();
        deleteAnr(context, anr);
    }

    private static void deleteAnr(Context context, @NonNull Anr anr) {
        DeleteCrashUtilsKt.deleteAnr(context, anr);
    }

    private static void logRateIsLimited() {
        InstabugSDKLogger.d(Constants.LOG_TAG, String.format(RateLimitedException.RATE_LIMIT_REACHED, Constants.FEATURE_NAME));
    }

    private static void uploadAnrLogs(final Anr anr) {

        InstabugSDKLogger.d(Constants.LOG_TAG, "START uploading all logs related to this ANR id = "
                + anr.getId());
        AnrsService.getInstance().uploadAnrLogs(anr, new Request.Callbacks<Boolean, Anr>() {

            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                try {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "ANR logs uploaded successfully");
                    anr.setAnrState(ATTACHMENTS_READY_TO_BE_UPLOADED);

                    //updating ANR in db
                    ContentValues contentValues = new ContentValues();
                    contentValues.put(COLUMN_ANR_UPLOAD_STATE, ATTACHMENTS_READY_TO_BE_UPLOADED);
                    AnrReportsDbHelper.update(anr.getId(), contentValues);


                    uploadAnrAttachments(anr);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error happened while uploading ANR: " + anr.getId() + "attachments.");
                }
            }

            @Override
            public void onFailed(Anr failedANR) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while uploading ANR logs");
            }
        });
    }

    private static void uploadAnrAttachments(final Anr anr) throws JSONException {

        InstabugSDKLogger.d(Constants.LOG_TAG, "Found " + anr.getAttachments().size() + " attachments related to" +
                " ANR: " + anr.getId());
        AnrsService.getInstance().uploadAnrAttachments(anr, new Request.Callbacks<Boolean, Anr>() {

            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Anr attachments uploaded successfully");

                Context context = Instabug.getApplicationContext();
                if (context != null) {
                    deleteAnrAndStateFile(context, anr);
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "unable to delete state file for ANR with id: " + anr.getId()
                            + "due to null context reference");
                }
                DiagnosticsLocator.getReporter().report(new CalibrationDiagnosticEvent(new ANRIncidentType(), Action.Synced));
            }

            @Override
            public void onFailed(Anr failedAnr) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while uploading ANR attachments");
            }
        });
    }

    public void start() {
        enqueueJob(IBGNetworkWorker.CRASH, new Runnable() {
            @Override
            public void run() {
                if (Instabug.getApplicationContext() != null) {
                    try {
                        uploadAnrs(Instabug.getApplicationContext());
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + e.getMessage() + " occurred while uploading ANRs", e);
                    }
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't upload ANRs due to null context");
                }
            }
        });
    }
}
