package com.instabug.anr

import android.content.Context
import android.os.Build
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.anr.cache.AnrReportsDbHelper
import com.instabug.anr.configuration.AnrConfigurationProvider
import com.instabug.anr.di.AnrServiceLocator
import com.instabug.anr.model.Anr
import com.instabug.anr.model.AnrState
import com.instabug.anr.network.InstabugAnrUploaderJob
import com.instabug.commons.PluginDelegate
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.crash.Constants
import com.instabug.commons.di.CommonsLocator
import com.instabug.crash.utils.CrashCleanupUtils
import com.instabug.library.Instabug
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.NetworkActivated
import com.instabug.library.util.AppUtils
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.visualusersteps.ReproCapturingProxy

class AnrPluginDelegate : PluginDelegate, AnrListener {

    private var anrDetectorThread: InstabugAnrDetectorThread? = null
    private lateinit var context: Context
    private val anrConfigurationHandler: ConfigurationsHandler by lazy {
        AnrServiceLocator.anrConfigurationHandler
    }
    private val anrConfigurationProvider: AnrConfigurationProvider by lazy {
        AnrServiceLocator.anrConfigurationProvider
    }
    private val reproProxy: ReproCapturingProxy by lazy {
        CommonsLocator.reproProxy
    }

    override fun init(context: Context) {
        this.context = context
        reproProxy.evaluate(anrConfigurationProvider)
        anrConfigurationHandler.migrateCurrentConfiguration()
    }

    override fun start(context: Context) {
        if (anrV2Enabled()) return
        PoolProvider.postIOTask {
            SettingsManager.getInstance()
                ?.reproConfigurations
                ?.let {
                    handleReproStateConfigurations(it.modesMap)
                }
            startUploaderServices()

        }
    }

    override fun wake() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Waking ANR plugin delegate")
        if (anrV2Enabled()) return
        startAnrDetectionIfPossible()
    }

    override fun sleep() {
        if (anrV2Enabled()) return
        anrDetectorThread?.interrupt()
        anrDetectorThread = null
    }

    override fun stop() {
        if (anrV2Enabled()) return
        anrDetectorThread = null
    }

    @WorkerThread
    private fun startUploaderServices() {
        if (isAnrEnabled() && AnrReportsDbHelper.getAnrsCount() > 0)
            InstabugAnrUploaderJob.getInstance().start()
    }


    override fun handleSDKCoreEvent(sdkCoreEvent: IBGSdkCoreEvent) {
        when (sdkCoreEvent) {
            NetworkActivated -> startAnrsUploaderService()
            is FeaturesFetched -> handleFeaturesFetched(sdkCoreEvent.response)
            is Features -> handleAnrStateChanged()
            is IBGSdkCoreEvent.ReproState -> handleReproStateConfigurations(sdkCoreEvent.modesMap)
            else -> {}
        }
    }

    private fun handleAnrStateChanged() {
        reproProxy.evaluate(anrConfigurationProvider)
        if (anrV2Enabled() || !isAnrEnabled()) {
            anrDetectorThread?.interrupt()
            anrDetectorThread = null

        } else
            startAnrDetectionIfPossible()
    }

    private fun anrV2Enabled() =
        Build.VERSION.SDK_INT >= Build.VERSION_CODES.R && anrConfigurationProvider.isAnrV2Enabled

    private fun handleFeaturesFetched(featuresResponse: String) {
        PoolProvider.postIOTask {
            anrConfigurationHandler.handleConfiguration(featuresResponse)
            handleAnrStateChanged()
        }
    }

    @VisibleForTesting
    fun getAnrDetectorThread(): InstabugAnrDetectorThread? {
        return anrDetectorThread
    }

    private fun startAnrDetection() {
        if (Instabug.isEnabled()) {
            val anrFactory = Anr.Factory()
            val processStateHelper = ProcessStateHelper()
            anrDetectorThread =
                InstabugAnrDetectorThread(this, anrFactory, processStateHelper)
            anrDetectorThread?.start()
        }
    }

    private fun startAnrsUploaderService() {
        if (anrConfigurationProvider.isAnrEnabled())
            PoolProvider.postIOTask {
                CrashCleanupUtils.cleanStaleANRStateFiles()
                val anrsCount = AnrReportsDbHelper.getAnrsCount()
                if (anrsCount > 0) {
                    InstabugAnrUploaderJob.getInstance().start()
                }
            }
    }

    private fun startAnrDetectionIfPossible() {
        if (canStartAnrDetection()) {
            startAnrDetection()
        }
    }

    private fun canStartAnrDetection(): Boolean {
        return anrDetectorThread == null && isAnrEnabled() && AppUtils.isAppForeground(Instabug.getApplicationContext())
    }

    private fun isAnrEnabled() = anrConfigurationProvider.isAnrEnabled()

    /**
     * This method is called from [InstabugAnrDetectorThread].
     **/
    @WorkerThread
    override fun onAnrDetected(anr: Anr) {
        anr.anrState = AnrState.READY_TO_BE_SENT
        AnrReportsDbHelper.insert(anr)
        // Upload the ANR
        InstabugAnrUploaderJob.getInstance().start()
    }

    private fun handleReproStateConfigurations(modesMap: Map<Int, Int>) {
        anrConfigurationHandler.handle(modesMap)
        reproProxy.evaluate(anrConfigurationProvider)
    }
}