package com.instabug.library.visualusersteps;

import static com.instabug.library.Instabug.getApplicationContext;
import static com.instabug.library.model.StepType.END_EDITING;
import static com.instabug.library.model.StepType.START_EDITING;
import static com.instabug.library.model.StepType.TAP;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.screenshot.analytics.AnalyticsEvent;
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;

import java.io.File;
import java.util.Deque;
import java.util.List;
import java.util.concurrent.LinkedBlockingDeque;

import io.reactivexport.Observable;
import io.reactivexport.schedulers.Schedulers;

/**
 * Created by tarek on 1/10/18.
 */

public class VisualUserSteps {

    private final Deque<Parent> parents;

    private int stepsCount;

    VisualUserSteps() {
        this.parents = new LinkedBlockingDeque<>();
    }

    Deque<Parent> getParents() {
        return parents;
    }

    @Nullable
    Parent getLastParent() {
        return parents.peekLast();
    }

    @Nullable
    private Parent getFirstParent() {
        return parents.peekFirst();
    }

    void addStep(Parent parent, VisualUserStep step) {
        // drop END_EDITING
        if (step.getStepType() != null && step.getStepType().equals(END_EDITING))
            return;

        if (isDuplicate(parent, step)) {
            return;
        }

        if (step.getStepType() != null && step.getStepType().equals(START_EDITING)) {
            if (step.getView() != null &&
                    !step.getView().equals(VisualUserStepsProvider.DEFAULT_TEXT_FIELD)) {
                step.setView(StringUtility.applyDoubleQuotations(step.getView()));
            }
        }

        parent.addStep(step);
        stepsCount++;
    }

    private boolean isDuplicate(Parent parent, VisualUserStep step) {
        if (parent != null) {
            VisualUserStep lastStep = parent.getLastStep();
            return lastStep != null
                    && step != null
                    && lastStep.getView() != null
                    && step.getView() != null
                    && lastStep.getView().replace("\"", "").equals(step.getView())
                    && lastStep.getStepType() != null
                    && lastStep.getStepType().equals(START_EDITING)
                    && lastStep.getScreenName() != null
                    && step.getScreenName() != null
                    && lastStep.getScreenName().equals(step.getScreenName());

        }
        return false;
    }

    void removeFirstStep() {
        Parent firstParent = getFirstParent();
        if (firstParent != null && firstParent.getStepsCount() > 1) {
            stepsCount--;
            if (getFirstParent() != null) {
                getFirstParent().removeFirst();
            }
        } else {
            removeFirstParent();
        }
    }

    int getStepsCount() {
        return stepsCount;
    }

    void addParent(Parent parent) {
        parents.add(parent);
    }

    int getParentsCount() {
        return parents.size();
    }

    private void removeFirstParent() {
        Parent parent = parents.peekFirst();
        if (parent != null) {
            Parent.Screenshot screenshot = parent.getScreenshot();
            // if has screenshot delete it
            if (screenshot != null) {
                deleteFileAsync(screenshot.getId());
                ScreenshotsAnalyticsEventBus.INSTANCE.post(AnalyticsEvent.ScreenshotEvent.ScreenshotsTrimmed.INSTANCE);
            }
            int firstParentStepsCount = parent.getStepsCount();
            stepsCount -= firstParentStepsCount;
            parents.pollFirst();
        }
    }

    void removeFirstParents(int count) {
        for (int i = 0; i < count; i++) {
            removeFirstParent();
        }
    }

    private boolean deleteFile(@Nullable final String fileName) {
        String message;
        Context context = getApplicationContext();
        if (fileName != null) {
            if (context != null) {
                File directory = CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
                String pathname = directory + File.separator + fileName;
                File file = new File(pathname);
                if (file.exists()) {
                    if (file.delete()) {
                        message = "VisualUserStep screenshot deleted! filename= " + fileName;
                        InstabugSDKLogger.v(Constants.LOG_TAG, message);
                        return true;
                    } else
                        message = "Couldn't delete screenshot=" + fileName + ". Something went wrong";
                } else {
                    int extensionIndex = FileUtils.getIndexOfExtension(fileName);
                    String extension = fileName.substring(extensionIndex);
                    String fileNameWithoutExtension = fileName.substring(0,
                            extensionIndex != -1 ? extensionIndex : fileName.length());
                    String encryptedFileName = directory + File.separator + fileNameWithoutExtension
                            + FileUtils.FLAG_ENCRYPTED + extension;
                    File encryptedFile = new File(encryptedFileName);
                    if (encryptedFile.exists()) {
                        if (encryptedFile.delete()) {
                            message = "VisualUserStep screenshot deleted! filename= " + encryptedFileName;
                            InstabugSDKLogger.v(Constants.LOG_TAG, message);
                            return true;
                        } else
                            message = "Couldn't delete screenshot=" + encryptedFileName + ". Something went wrong";
                    } else message = "Couldn't execute deleteFile(). File does not exist";
                }
            } else message = "Couldn't execute deleteFile(). Context is null";
        } else message = "Couldn't execute deleteFile(). file name is null";
        InstabugSDKLogger.e(Constants.LOG_TAG, message);
        return false;
    }

    @SuppressLint("CheckResult")
    public void cleanDirectoryAsync() {
        final String[] message = new String[1];
        Observable.fromCallable(() -> {
                    Context context = getApplicationContext();
                    if (context != null) {
                        File directory = CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
                        if (directory != null) {
                            if (directory.exists()) {
                                String[] list = directory.list();
                                if (list != null) {
                                    for (String file : list) {
                                        deleteFile(file);
                                    }
                                }
                                if (directory.delete()) {
                                    message[0] = "VisualUserStep screenshot directory {" + directory + "} deleted";
                                    InstabugSDKLogger.v(Constants.LOG_TAG, message[0]);
                                    return true;
                                } else {
                                    message[0] = "Couldn't delete directory " + directory + ". Something went wrong";
                                }
                            } else {
                                message[0] = "Couldn't execute deleteFile(). Directory does not exist";
                            }
                        } else {
                            message[0] = "Couldn't execute deleteFile(). directory is null";
                        }
                    } else {
                        message[0] = "Couldn't execute deleteFile(). Context is null";
                    }
                    return false;
                }).subscribeOn(Schedulers.io())
                .subscribe(deleted -> {
                    if (!deleted) {
                        InstabugSDKLogger.i(Constants.LOG_TAG, message[0]);
                    } else {
                        InstabugSDKLogger.d(Constants.LOG_TAG, message[0]);
                    }
                });
    }

    @SuppressLint("CheckResult")
    private void deleteFileAsync(@Nullable final String fileName) {
        getDeleteFileObservable(fileName).subscribeOn(Schedulers.io())
                .subscribe(aBoolean -> {
                });
    }

    private Observable<Boolean> getDeleteFileObservable(@Nullable final String fileName) {
        return Observable.fromCallable(() -> deleteFile(fileName));
    }

    public void removeAllSteps() {
        parents.clear();
    }


    void removeLastTapStep() {
        if (getLastParent() != null && getLastParent().getSteps().size() > 0) {
            VisualUserStep last = getLastParent().getSteps().getLast();
            if (last.getStepType() != null && last.getStepType().equals(TAP)) {
                getLastParent().removeLast();
                stepsCount--;
            }
        }
    }

    public void removeIrrelevantSteps(Parent parent, List<VisualUserStep> stepsToRemove) {
        parent.getSteps().removeAll(stepsToRemove);
        stepsCount -= stepsToRemove.size();
    }

    void removeParent(@NonNull Parent parent){
        parents.remove(parent);
    }
}
