package com.instabug.library.visualusersteps

import com.instabug.library.Constants
import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.InstabugFeaturesManager
import com.instabug.library.core.InstabugCore
import com.instabug.library.model.StepType
import com.instabug.library.model.UserStep
import com.instabug.library.tracking.InstabugTrackingStepsProvider
import com.instabug.library.usersteps.UserStepsMessageGenerator
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.OrderedExecutorService

interface CPUserStepsHandler {
    fun addUserStep(
        timestamp: Long,
        @StepType stepType: String,
        message: String,
        label: String?,
        viewType: String,
    )
}
private const val VIEW_TYPE_LENGTH = 30
class CPUserStepsHandlerImpl(
    private val executor: OrderedExecutorService,
    private val reproStepsProxy: ReproStepsCapturingProxy
) : CPUserStepsHandler {
    override fun addUserStep(
        timestamp: Long,
        @StepType stepType: String,
        message: String,
        label: String?,
        viewType: String,
    ) {
        operateOnExecutor {
            reproStepsProxy.addVisualUserStep(
                stepType,
                InstabugCore.getCurrentView(),
                message,
                null,
            )

            if (shouldReportUserStep()) {
                InstabugTrackingStepsProvider.getInstance().addUserStep(
                    UserStep().apply {
                        timeStamp = timestamp
                        setMessage(getUserStepsMessage(stepType, viewType, label))
                        setType(stepType)
                        args = UserStep.Args(
                            type,
                            label,
                            trimViewType(viewType),
                            InstabugCore.getCurrentView()
                        )
                    },
                )
            }
        }
    }

    private fun shouldReportUserStep() =
        InstabugFeaturesManager.getInstance()
            .getFeatureState(IBGFeature.TRACK_USER_STEPS) == Feature.State.ENABLED

    private fun getUserStepsMessage(
        @StepType stepType: String,
        viewType: String,
        label: String?,
    ): String =
        UserStepsMessageGenerator.generateUserActionStepMessage(
            stepType,
            trimViewType(viewType),
            null,
            label,
            InstabugCore.getCurrentView(),
        )

    private fun operateOnExecutor(runnable: Runnable) {
        executor.execute(Constants.USER_STEPS_EXECUTOR_KEY, runnable)
    }

    fun trimViewType(viewType: String): String {
        return if (viewType.length > VIEW_TYPE_LENGTH) {
            InstabugSDKLogger.w(
                Constants.LOG_TAG,
                "User steps View Type string exceed the 30 characters limit. Extra characters will be trimmed"
            )
            viewType.take(VIEW_TYPE_LENGTH)
        } else {
            viewType
        }
    }
}

