package com.instabug.library.tracking;

import android.app.Activity;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.google.android.material.tabs.TabLayout;
import com.instabug.library.util.WeakReferencePair;
import com.instabug.library.visualusersteps.InstabugOnTabSelectedListener;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class NavigableViewsTrackingDelegate {

    @NonNull
    private final Map<String, List<WeakReferencePair<TabLayout, InstabugOnTabSelectedListener>>> tabLayoutMap;

    public NavigableViewsTrackingDelegate() {
        this.tabLayoutMap = new HashMap<>();
    }

    void onActivityResumed(@NonNull Activity activity) {
        View rootView = getRootView(activity);
        List<TabLayout> tabLayouts = getTabLayout(rootView);
        if (tabLayouts != null && tabLayouts.size() > 0)
            subscribeTo(tabLayouts, activity.getClass().getName());
    }

    void onActivityPaused(@NonNull Activity activity) {
        unsubscribeFrom(activity.getClass().getName());
    }

    void onFragmentViewCreated(@NonNull Fragment fragment) {
        View rootView = getRootView(fragment);
        if (rootView != null) {
            List<TabLayout> tabLayouts = getTabLayout(rootView);
            if (tabLayouts != null && tabLayouts.size() > 0)
                subscribeTo(tabLayouts, fragment.getClass().getName());
        }
    }

    void onFragmentDetached(@NonNull Fragment fragment) {
        unsubscribeFrom(fragment.getClass().getName());
    }

    @NonNull
    private View getRootView(@NonNull Activity activity) {
        return activity.getWindow().getDecorView();
    }

    @Nullable
    private View getRootView(@NonNull Fragment fragment) {
        return fragment.getView();
    }

    @Nullable
    private List<TabLayout> getTabLayout(@Nullable View rootView) {
        List<TabLayout> tabLayouts = new ArrayList<>();
        if (rootView instanceof TabLayout) {
            tabLayouts.add((TabLayout) rootView);
            return tabLayouts;
        } else if (rootView instanceof ViewGroup) {
            ViewGroup group = (ViewGroup) rootView;
            for (int i = 0; i < group.getChildCount(); i++) {
                View child = group.getChildAt(i);
                if (child instanceof TabLayout)
                    tabLayouts.add((TabLayout) child);
                else if (child instanceof ViewGroup) {
                    List<TabLayout> tabLayout = getTabLayout(child);
                    if (tabLayout != null)
                        tabLayouts.addAll(tabLayout);
                }
            }
            return tabLayouts;
        }
        return null;
    }

    private void subscribeTo(@NonNull List<TabLayout> tabLayouts, @NonNull String tag) {
        for (TabLayout tablayout : tabLayouts) {
            if (!isRegistered(tablayout)) {
                // get last segment (eg. android.view.View -> View)
                String screenName = tag.substring(tag.lastIndexOf(".") + 1);
                InstabugOnTabSelectedListener listener = new InstabugOnTabSelectedListener(screenName);
                tablayout.addOnTabSelectedListener(listener);

                List<WeakReferencePair<TabLayout, InstabugOnTabSelectedListener>> layoutListenerPairs = tabLayoutMap.get(tag);
                if (layoutListenerPairs == null) {
                    layoutListenerPairs = new ArrayList<>();
                    layoutListenerPairs.add(new WeakReferencePair<>(tablayout, listener));
                    tabLayoutMap.put(tag, layoutListenerPairs);
                } else {
                    layoutListenerPairs.add(new WeakReferencePair<>(tablayout, listener));
                }
            }
        }
    }

    private void unsubscribeFrom(@NonNull String name) {
        List<WeakReferencePair<TabLayout, InstabugOnTabSelectedListener>> layoutListenerPairs = tabLayoutMap.get(name);
        if (layoutListenerPairs != null && layoutListenerPairs.size() > 0)
            for (WeakReferencePair<TabLayout, InstabugOnTabSelectedListener> listenerLayoutPair : layoutListenerPairs) {
                if (listenerLayoutPair.first != null && listenerLayoutPair.second != null) {
                    TabLayout tabLayout = listenerLayoutPair.first.get();
                    InstabugOnTabSelectedListener listener = listenerLayoutPair.second.get();
                    if (tabLayout != null && listener != null) {
                        tabLayout.removeOnTabSelectedListener(listener);
                    }
                }
            }
    }

    private boolean isRegistered(@NonNull TabLayout tabLayout) {
        for (List<WeakReferencePair<TabLayout, InstabugOnTabSelectedListener>> tabLayoutWeakReferenceList : tabLayoutMap.values()) {
            for (WeakReferencePair<TabLayout, InstabugOnTabSelectedListener> tabLayoutWeakReference : tabLayoutWeakReferenceList) {
                TabLayout registeredTabLayout = tabLayoutWeakReference.first.get();
                if (registeredTabLayout != null && registeredTabLayout.equals(tabLayout))
                    return true;
            }
        }
        return false;
    }
}
