package com.instabug.library.screenshot;

import static com.instabug.library.model.ScreenshotCaptureStatus.FAILURE;
import static com.instabug.library.model.ScreenshotCaptureStatus.SUCCESS;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.media.projection.MediaProjectionManager;
import android.os.Handler;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.UiThread;

import com.instabug.library.Constants;
import com.instabug.library.core.eventbus.ScreenCaptureEventBus;
import com.instabug.library.model.ScreenshotResponse;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;

import io.reactivexport.functions.Consumer;

/**
 * @author hossam.
 */
@TargetApi(21)
public class ScreenshotManager implements Consumer<ScreenshotResponse> {
    @Nullable
    private ScreenshotCaptor.CapturingCallback screenshotCapturingListener;

    public static final ScreenshotManager INSTANCE = new ScreenshotManager();

    @Nullable
    private Intent mIntent;

    private ScreenshotManager() {
        ScreenCaptureEventBus.getInstance().subscribe(this);
    }

    public static ScreenshotManager getInstance() {
        return INSTANCE;
    }

    public void requestScreenshotPermission(@NonNull Activity activity, int requestId) {
        MediaProjectionManager mediaProjectionManager = (MediaProjectionManager) activity.getSystemService(Context.MEDIA_PROJECTION_SERVICE);
        activity.startActivityForResult(mediaProjectionManager.createScreenCaptureIntent(), requestId);
    }


    public void onActivityResult(int resultCode, @Nullable Intent data, final boolean isInitial,
                                 @Nullable final ScreenshotCaptor.CapturingCallback screenshotCapturingListener) {
        if (resultCode == Activity.RESULT_OK && data != null) {
            mIntent = data;
        } else mIntent = null;
        if (isInitial && screenshotCapturingListener != null) {
            captureInitialScreenShot(screenshotCapturingListener);
        }

    }

    private void captureInitialScreenShot(final ScreenshotCaptor.CapturingCallback screenshotCapturingListener) {
        final Handler screenshotHandler = new Handler();
        final Runnable screenshotRunnable = new Runnable() {
            @Override
            public void run() {
                takeScreenshot(new ScreenshotCaptor.CapturingCallback() {
                    @Override
                    public void onCapturingSuccess(final Bitmap bitmap) {
                        if (screenshotCapturingListener != null) {
                            screenshotCapturingListener.onCapturingSuccess(bitmap);

                        }

                    }

                    @Override
                    public void onCapturingFailure(Throwable throwable) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "initial screenshot capturing got error: " + throwable.getMessage(), throwable);
                        if (screenshotCapturingListener != null) {
                            screenshotCapturingListener.onCapturingFailure(throwable);
                            SettingsManager.getInstance().setProcessingForeground(false);
                        }
                    }
                });
            }
        };
        screenshotHandler.postDelayed(screenshotRunnable, 500);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void saveExtraScreenshotBitmap(Bitmap bitmap, Handler screenshotHandler,
                                           ScreenshotCaptor.CapturingCallback screenshotCapturingListener) {
        if (screenshotCapturingListener != null) {
            screenshotCapturingListener.onCapturingSuccess(bitmap);
            SettingsManager.getInstance().setProcessingForeground(false);
            screenshotHandler.removeCallbacksAndMessages(null);
        }
    }

    @UiThread
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void takeScreenshot(@NonNull final ScreenshotCaptor.CapturingCallback screenshotCapturingListener) {
        this.screenshotCapturingListener = screenshotCapturingListener;
        Activity currentActivity =
                InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (currentActivity != null) {
            currentActivity.startService(ScreenshotCaptureService.newIntent(currentActivity, mIntent));
        }
    }

    @Override
    public void accept(ScreenshotResponse screenshotResponse) {
        if (screenshotCapturingListener != null) {
            switch (screenshotResponse.getStatus()) {
                case SUCCESS:
                    if (screenshotResponse.getScreenshotBitmap() != null) {
                        screenshotCapturingListener.onCapturingSuccess(screenshotResponse.getScreenshotBitmap());
                    }
                    break;
                case FAILURE:
                    if (screenshotResponse.getThrowable() != null) {
                        screenshotCapturingListener.onCapturingFailure(screenshotResponse.getThrowable());
                    }
                    break;
                default:
                    break;
            }
        }
    }
}
