package com.instabug.library.networkv2.service;


import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.BuildConfig;
import com.instabug.library.Constants;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.model.FeaturesCache;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TaskDebouncer;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.TimeUnit;

/**
 * Created by mesbah on 12/28/2015.
 */
public class FeaturesService {
    private static volatile FeaturesService INSTANCE;
    private final TaskDebouncer networkTaskDebouncer;
    private final NetworkManager networkManager;

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of FeaturesService
     */
    public synchronized static FeaturesService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new FeaturesService();
        }
        return INSTANCE;
    }

    private FeaturesService() {
        networkManager = new NetworkManager();
        networkTaskDebouncer = new TaskDebouncer(TimeUnit.SECONDS.toMillis(2));
    }

    public void getAppFeatures(final Request.Callbacks<String, Throwable> callbacks) {
        networkTaskDebouncer.debounce(() -> PoolProvider.postIOTask(() -> fetch(callbacks)));
    }

    @VisibleForTesting
    Request buildAppFeaturesRequest() throws JSONException {
        Request.Builder requestBuilder = new Request.Builder()
                .endpoint(Endpoints.APP_SETTINGS)
                .tokenProvider(() -> SettingsManager.getInstance().getAppToken())
                .method(RequestMethod.GET);

        FeaturesCache featuresCache = SettingsManager.getInstance().getFeaturesCache();
        if (featuresCache != null && featuresCache.getHash() != null) {
            String featuresHash = featuresCache.getHash();
            if (featuresHash != null) {
                requestBuilder.addHeader(new RequestParameter<>(Header.IF_MATCH, featuresHash));
            }
        }
        return requestBuilder.build();
    }

    @VisibleForTesting
    NetworkManager getNetworkManager() {
        return networkManager;
    }

    private void fetch(final Request.Callbacks<String, Throwable> callbacks) {
        try {
            if (callbacks != null) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Getting enabled features for this application");
                Request getFeaturesRequest = buildAppFeaturesRequest();
                networkManager.doRequest(IBGNetworkWorker.CORE, RequestType.NORMAL, getFeaturesRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                    @Override
                    public void onSucceeded(RequestResponse requestResponse) {
                        if (requestResponse != null) {
                            String extractedJson = extractJson(requestResponse);
                            if (extractedJson != null) {
                                InstabugSDKLogger.d(Constants.LOG_TAG, "getAppFeatures request completed");
                                InstabugSDKLogger.v(Constants.LOG_TAG, "Features response: " + requestResponse);
                                callbacks.onSucceeded(extractedJson);
                            }
                        }
                    }

                    @Override
                    public void onFailed(Throwable error) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "getAppFeatures request got error: " + error.getMessage());
                        InstabugCore.reportError(error, "Failed to cache features settings due to: " + error.getMessage());
                        callbacks.onFailed(error);
                    }
                });
            }
        } catch (JSONException e) {
            callbacks.onFailed(e);
        }
    }

    /**
     * @return response body in a json format if the request was successful or null otherwise
     */
    @Nullable
    private String extractJson(RequestResponse requestResponse) {
        int code = requestResponse.getResponseCode();
        switch (code) {
            case RequestResponse.HttpStatusCode._2xx.OK:
                String responseBody = (String) requestResponse.getResponseBody();
                try {
                    long ttl = 0;
                    boolean isActive = true;
                    if (responseBody != null) {
                        JSONObject responseJson = new JSONObject(responseBody);
                        ttl = responseJson.optLong("ttl", 0);
                        isActive = responseJson.optBoolean("is_active", true);
                    }
                    String hash = requestResponse.getHeaders().get(Header.IF_MATCH);
                    FeaturesCache featuresCache = new FeaturesCache(ttl, isActive, BuildConfig.SDK_VERSION, hash);
                    SettingsManager.getInstance().setFeaturesCache(featuresCache);
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to cache features settings due to: " + e.getMessage());
                }
                return responseBody;
              case RequestResponse.HttpStatusCode._3xx.NOT_MODIFIED:
                InstabugSDKLogger.d(Constants.LOG_TAG, "Features list did not get modified. Moving on...");
                IBGCoreEventPublisher.post(IBGSdkCoreEvent.Features.Fetched.INSTANCE);
                return null;
                //modified
              case RequestResponse.HttpStatusCode._4xx.BAD_REQUEST:

            default:
                InstabugSDKLogger.d(Constants.LOG_TAG, "Caught unhandled case with code (" + code + ")");
                return null;
        }
    }
}
