package com.instabug.library.networkinterception.sanitization

import android.net.Uri
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.model.NetworkClient
import com.instabug.library.networkinterception.model.NetworkLogModel
import com.instabug.library.sanitize.Sanitizer
import com.instabug.library.util.extenstions.runOrLogError

class InstabugQueryParamsSanitizer(
    private val instabugDomains: Set<String>,
    private val configurations: IBGNetworkInterceptionConfigurationProvider
) : Sanitizer<NetworkLogModel.Builder> {

    private val sanitizationQueries: Set<String>
        get() = configurations.v2InstabugSanitizationQueryParams

    override fun sanitize(item: NetworkLogModel.Builder): NetworkLogModel.Builder =
        item.also { builder ->
            builder.takeIf { it.clientId == NetworkClient.URL_CONNECTION }
                ?.takeUnless { sanitizationQueries.isEmpty() }
                ?.applySanitization()
        }

    private fun NetworkLogModel.Builder.applySanitization() = apply {
        runOrLogError {
            url.let { Uri.parse(it) }?.takeIf { it.hasInstabugDomain() }
        }.getOrNull()?.sanitize()?.toString()?.also { setUrl(it) }
    }

    private fun Uri.sanitize(): Uri =
        buildUpon()
            .clearQuery()
            .apply {
                val keywords = sanitizationQueries.map { it.lowercase() }
                queryParameterNames.asSequence()
                    .filterNot { it.lowercase() in keywords }
                    .forEach { appendQueryParameter(it, getQueryParameter(it)) }
            }.build()

    private fun Uri.hasInstabugDomain(): Boolean =
        instabugDomains.any { host.equals(it, true) }
}
