package com.instabug.library.networkinterception.sanitization

import android.net.Uri
import com.instabug.library.Constants
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.model.NetworkLogModel
import com.instabug.library.sanitize.Sanitizer
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.collections.toQueryString
import com.instabug.library.util.extenstions.extractQueryParams

class AutoMaskingSanitizerV2(
    private val autoMaskingHeaderKeys: Set<String>,
    private val autoMaskingQueryKeys: Set<String>,
    private val configurationProvider: IBGNetworkInterceptionConfigurationProvider
) : Sanitizer<NetworkLogModel.Builder> {

    val MASKED_VALUE = "*****"

    override fun sanitize(item: NetworkLogModel.Builder): NetworkLogModel.Builder =
        runCatching {
            item.takeIf { configurationProvider.isAutoMaskingEnabled }
                ?.apply {
                    sanitizeHeaders(this)
                    sanitizeQueryParams(this)
                } ?: item
        }.getOrElse {
            InstabugSDKLogger.e(Constants.LOG_TAG, it.message.orEmpty(), it)
            item
        }

    private fun sanitizeHeaders(item: NetworkLogModel.Builder) {
        val requestHeaders = item.requestBuilder?.headers?.toMutableMap()
        val responseHeaders = item.responseBuilder?.headers?.toMutableMap()

        requestHeaders?.forEach { entry ->
            if (shouldSanitizeHeader(entry.key)) {
                requestHeaders[entry.key] = MASKED_VALUE
            }
        }

        responseHeaders?.forEach { entry ->
            if (shouldSanitizeHeader(entry.key)) {
                responseHeaders[entry.key] = MASKED_VALUE
            }
        }

        item.requestBuilder?.setHeaders(requestHeaders?.toMap())
        item.responseBuilder?.setHeaders(responseHeaders?.toMap())
    }

    private fun sanitizeQueryParams(item: NetworkLogModel.Builder) {
        item.url.let { url ->
            val queryParams = url.extractQueryParams()?.toMutableMap()

            val uriBuilder = Uri.parse(url).buildUpon()

            queryParams?.forEach { query ->
                if (shouldSanitizeQuery(query.key)) {
                    queryParams[query.key] = MASKED_VALUE
                }
            }

            if (queryParams?.isNotEmpty() == true) {
                uriBuilder.encodedQuery(queryParams.toQueryString())
                item.setUrl(
                    uriBuilder.build().toString()
                )
            }
        }
    }

    private fun shouldSanitizeHeader(key: String): Boolean =
        autoMaskingHeaderKeys.contains(key.lowercase())

    private fun shouldSanitizeQuery(key: String): Boolean =
        autoMaskingQueryKeys.contains(key.lowercase())
}
