package com.instabug.library.networkinterception.model.map

import com.instabug.library.map.Mapper
import com.instabug.library.networkinterception.model.NetworkLogModel
import com.instabug.library.networkinterception.model.NetworkLogRequestModel
import com.instabug.library.networkinterception.model.NetworkLogResponseModel
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.toJsonObject
import org.json.JSONObject
import java.util.concurrent.TimeUnit

open class CoreCommonNetworkLogJsonMapper : Mapper<NetworkLogModel, JSONObject?> {

    override fun map(from: NetworkLogModel): JSONObject? =
        runCatching {
            mapToJson(from)
        }.getOrLogAndReport(null, FAILED_TO_PARSE_NETWORK_LOG_TO_JSON, true)

    protected open fun mapToJson(from: NetworkLogModel): JSONObject =
        from.run {
            JSONObject().also { jsonObject ->
                jsonObject.put(KEY_URL, url)
                jsonObject.put(KEY_DATE, "$startTimestampMs")
                jsonObject.put(KEY_NETWORK_TIME, TimeUnit.MICROSECONDS.toMillis(durationMus))
                jsonObject.put(KEY_USER_MODIFIED, userModified)

                request?.let { mapRequest(it, jsonObject) }
                response?.let { mapResponse(it, jsonObject) }
            }
        }

    private fun mapRequest(
        request: NetworkLogRequestModel,
        jsonObject: JSONObject
    ) {
        jsonObject.put(KEY_METHOD, request.method)
        request.headers
            ?.toJsonObject()
            ?.apply { request.contentLength?.let { put(KEY_CONTENT_LENGTH, it) } }
            ?.let { jsonObject.put(KEY_REQUEST_HEADERS, it) }
        request.body
            ?.bodyStringRepresentation()
            ?.let { jsonObject.put(KEY_REQUEST, it) }
    }

    private fun mapResponse(
        response: NetworkLogResponseModel,
        jsonObject: JSONObject
    ) {
        jsonObject.put(KEY_RESPONSE_CODE, response.responseCode)
        response.headers?.toJsonObject()
            ?.let { jsonObject.put(KEY_RESPONSE_HEADERS, it) }
        response.clientSideThrowable?.let { it.message ?: it.javaClass.simpleName }
            ?.also { jsonObject.put(KEY_RESPONSE, it) }
            ?: response.body
                ?.bodyStringRepresentation()
                ?.let { jsonObject.put(KEY_RESPONSE, it) }
    }

    private fun String.bodyStringRepresentation() =
        runCatching { JSONObject(this) }.getOrDefault(this)

    companion object {

        const val FAILED_TO_PARSE_NETWORK_LOG_TO_JSON: String =
            "Failed to parse Network Log to JSON:"
        const val KEY_DATE = "date"
        const val KEY_URL = "url"
        const val KEY_REQUEST = "request"
        const val KEY_RESPONSE = "response"
        const val KEY_METHOD = "method"
        const val KEY_RESPONSE_CODE = "status"
        const val KEY_REQUEST_HEADERS = "headers"
        const val KEY_RESPONSE_HEADERS = "response_headers"
        const val KEY_NETWORK_TIME = "response_time"
        const val KEY_USER_MODIFIED = "user_modified"
        const val KEY_CONTENT_LENGTH = "Content-Length"
    }
}
