package com.instabug.library.networkinterception.externaltrace

import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.internal.servicelocator.Provider
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.model.NetworkLogW3CExternalTraceIdInfo

class W3CExternalNetworkTraceIdFactory(
    private val configProvider: IBGNetworkInterceptionConfigurationProvider,
    private val formatProvider: Provider<W3CFormatDetails>
) : ParameterizedFactory<NetworkLogW3CExternalTraceIdInfo?, Map<String, String>?> {

    override fun create(type: Map<String, String>?): NetworkLogW3CExternalTraceIdInfo? {
        if (!configProvider.isW3CNetworkExternalTraceIdEnabled()) return null
        return type?.externalTraceValue()?.let {
            NetworkLogW3CExternalTraceIdInfo(
                isCaptured = true,
                syncableCapturedTraceId = it.takeIf { configProvider.isAttachingCapturedW3CExternalTraceIdEnabled }
            )
        } ?: generateTraceId()
    }

    private fun generateTraceId(): NetworkLogW3CExternalTraceIdInfo {
        val generatedW3CFormat = formatProvider()
        return NetworkLogW3CExternalTraceIdInfo(
            isCaptured = false,
            pid = generatedW3CFormat.pid,
            timestampSeconds = generatedW3CFormat.timestampSeconds,
            fullyGeneratedId = generatedW3CFormat.traceFormat,
            syncableGeneratedTraceId = generatedW3CFormat.traceFormat
                .takeIf { configProvider.isAttachingGeneratedW3CExternalTraceIdEnabled },
            headers = mapOf(TraceParentHeaderKey to generatedW3CFormat.traceFormat)
        )
    }

    private fun Map<String, String>.externalTraceValue() =
        get(TraceParentHeaderKey)

    companion object {
        const val TraceParentHeaderKey = "traceparent"
    }
}
