package com.instabug.library.networkinterception.externaltrace

import com.instabug.library.internal.servicelocator.Provider
import java.util.Random
import java.util.concurrent.TimeUnit
import kotlin.math.abs

internal class TimestampProviderSeconds : Provider<Long> {
    override fun invoke(): Long =
        System.currentTimeMillis().let(TimeUnit.MILLISECONDS::toSeconds)
}

internal class RandomUIntProvider : Provider<Long> {
    override fun invoke(): Long =
        Random().nextInt().toUInt().let { if (it == 0u) it + 1u else it }.toLong()
}

data class W3CFormatDetails(
    val pid: Long,
    val timestampSeconds: Long,
    val traceFormat: String
)

internal class W3CFormatProvider(
    private val timestampProvider: Provider<Long>,
    private val randomIntProvider: Provider<Long>
) : Provider<W3CFormatDetails> {
    override fun invoke(): W3CFormatDetails {
        val timestampSeconds = timestampProvider()
        val pid = randomIntProvider.invoke()
        val pidHex = pid.toString(HexRadix).to0Prefixed8CharHex()
        val traceId = W3CTraceIdFormat
            .replace("{timestamp}", timestampSeconds.toString(HexRadix).to0Prefixed8CharHex())
            .replace("{pid}", pidHex)
        val parentId = W3CParentIdFormat.replace("{pid}", pidHex)
        val traceHeader = W3CHeaderFormat
            .replace("{trace-id}", traceId)
            .replace("{parent-id}", parentId)
        return W3CFormatDetails(
            pid = pid,
            timestampSeconds = timestampSeconds,
            traceFormat = traceHeader
        )
    }

    private fun String.to0Prefixed8CharHex(): String {
        val delta = RequiredHexDigits - this.length
        return when {
            delta > 0 -> buildString {
                repeat(delta) { append("0") }
                append(this@to0Prefixed8CharHex)
            }
            delta < 0 -> substring(abs(delta))
            else -> this
        }
    }

    companion object {
        private const val HexRadix = 16
        private const val RequiredHexDigits = 8
        private const val IBGConstant = "4942472d"
        private const val W3CTraceIdFormat = "{timestamp}{pid}{timestamp}{pid}"
        private const val W3CParentIdFormat = "$IBGConstant{pid}"
        private const val W3CHeaderFormat = "00-{trace-id}-{parent-id}-01"
    }
}
