package com.instabug.library.networkinterception.delegate.validate

import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.util.lcIsContentEncoded
import com.instabug.library.networkinterception.util.lcIsFileDownload
import com.instabug.library.networkinterception.util.lcIsMultiPart
import com.instabug.library.networkinterception.util.lcIsOctetStream
import com.instabug.library.networkinterception.util.lcIsProtobuf
import com.instabug.library.networkinterception.util.toLowerCaseKeys

open class DefaultResponseBodyValidator(
    protected val configurations: IBGNetworkInterceptionConfigurationProvider
) : NetworkLogBodyValidator {

    private val maxAllowedBytes: Long
        get() = configurations.v2MaxAllowedBodySizeBytes

    override fun invoke(headers: Map<String, String>, contentLength: Long): Pair<Boolean, String?> =
        validate(headers.toLowerCaseKeys(), contentLength)

    protected open fun validate(
        lowerCaseKeyHeaders: Map<String, String>,
        contentLength: Long
    ) = when {
        contentLength > maxAllowedBytes -> false to generateLargeBodyMessage(maxAllowedBytes)
        shouldSkipRequestBodyCapture() -> false to null
        lowerCaseKeyHeaders.lcIsOctetStream() -> false to BINARY_RAW_DATA
        lowerCaseKeyHeaders.lcIsFileDownload() -> false to BINARY_RAW_DATA
        lowerCaseKeyHeaders.encodingNotAllowed() -> false to null
        lowerCaseKeyHeaders.lcIsMultiPart() -> false to MULTIPART_BODY
        lowerCaseKeyHeaders.lcIsProtobuf() -> false to PROTOBUF_RESPONSE_BODY
        else -> true to null
    }
    private fun shouldSkipRequestBodyCapture() = !configurations.shouldCaptureNetworkBodyLogs
    protected open fun Map<String, String>.encodingNotAllowed() = lcIsContentEncoded()
}
