package com.instabug.library.networkinterception.delegate.extract

import android.os.Build
import androidx.annotation.RequiresApi
import java.io.InputStream
import java.io.OutputStream

class InputStreamWrapper(
    private val delegate: InputStream,
    private val plaintTextOutputStream: PlainTextByteArrayOutputStream,
    private val onBodyReady: (body: String?, length: Long?) -> Unit
) : InputStream() {

    override fun close() = delegate.close().also { dispatchResult() }

    override fun read(): Int = delegate.read()
        .also { plaintTextOutputStream.write(it) }

    override fun read(b: ByteArray?): Int = delegate.read(b)
        .also { readCount ->
            b?.let { plaintTextOutputStream.write(it, 0, readCount) }
        }

    override fun read(b: ByteArray?, off: Int, len: Int): Int = delegate.read(b, off, len)
        .also { readCount ->
            b?.let { plaintTextOutputStream.write(it, off, readCount) }
        }

    @Suppress("Since15")
    @RequiresApi(Build.VERSION_CODES.TIRAMISU)
    override fun readAllBytes(): ByteArray = delegate.readAllBytes()
        .also { plaintTextOutputStream.write(it) }

    @Suppress("Since15")
    @RequiresApi(Build.VERSION_CODES.TIRAMISU)
    override fun readNBytes(len: Int): ByteArray = delegate.readNBytes(len)
        .also { plaintTextOutputStream.write(it) }

    @Suppress("Since15")
    @RequiresApi(Build.VERSION_CODES.TIRAMISU)
    override fun readNBytes(b: ByteArray?, off: Int, len: Int): Int =
        delegate.readNBytes(b, off, len)
            .also { readCount ->
                b?.let { plaintTextOutputStream.write(it, off, readCount) }
            }

    override fun skip(n: Long): Long = delegate.skip(n)

    @Suppress("Since15")
    @RequiresApi(Build.VERSION_CODES.UPSIDE_DOWN_CAKE)
    override fun skipNBytes(n: Long) = delegate.skipNBytes(n)

    override fun available(): Int = delegate.available()

    override fun mark(readlimit: Int) {
        runCatching {
            delegate.mark(readlimit).also {
                plaintTextOutputStream.mark()
            }
        }.onFailure { plaintTextOutputStream.unmark() }.getOrThrow()
    }

    override fun reset() {
        runCatching {
            delegate.reset().also {
                plaintTextOutputStream.resetToMark()
            }
        }.onFailure { plaintTextOutputStream.unmark() }.getOrThrow()
    }

    override fun markSupported(): Boolean = delegate.markSupported()

    @Suppress("Since15")
    @RequiresApi(Build.VERSION_CODES.TIRAMISU)
    override fun transferTo(out: OutputStream?): Long = delegate.transferTo(out)

    private fun dispatchResult() {
        onBodyReady(
            plaintTextOutputStream.generateBodyText(),
            plaintTextOutputStream.byteCount
        )
    }
}
