package com.instabug.library.networkinterception.delegate

import com.instabug.library.map.Mapper
import com.instabug.library.networkinterception.delegate.extract.NetworkBodyExtractor
import com.instabug.library.networkinterception.delegate.validate.NetworkLogBodyValidator
import com.instabug.library.networkinterception.invalidate.HeadersInvalidator
import com.instabug.library.networkinterception.model.NetworkLogModel

class DefaultNetworkInterceptorDelegate<REQUEST, RESPONSE>(
    private val graphQlQueryMapper: Mapper<Map<String, String>, String?>?,
    private val requestHeadersInvalidators: HeadersInvalidator,
    private val requestBodyCapturingValidator: NetworkLogBodyValidator?,
    private val responseBodyCapturingValidator: NetworkLogBodyValidator?,
    private val requestBodyExtractor: NetworkBodyExtractor<REQUEST>?,
    private val responseBodyExtractor: NetworkBodyExtractor<RESPONSE>?
) : NetworkInterceptorDelegate<REQUEST, RESPONSE> {

    override fun getGraphQLQuery(requestHeaders: Map<String, String>): String? =
        graphQlQueryMapper?.map(requestHeaders)

    override fun invalidateRequestHeaders(
        requestHeaders: MutableMap<String, String>?,
        networkLogBuilder: NetworkLogModel.Builder
    ): MutableMap<String, String>? = requestHeadersInvalidators(requestHeaders, networkLogBuilder)

    override fun consumeRequestBody(
        requestHeaders: Map<String, String>,
        contentLength: Long,
        bodyFactory: () -> REQUEST,
        onBodyReady: (body: String?, length: Long?) -> Unit
    ): REQUEST? = requestBodyExtractor?.validateAndExtract(
        requestHeaders,
        contentLength,
        requestBodyCapturingValidator,
        bodyFactory,
        onBodyReady
    )

    override fun consumeResponseBody(
        responseHeaders: Map<String, String>,
        contentLength: Long,
        bodyFactory: () -> RESPONSE,
        onBodyReady: (body: String?, length: Long?) -> Unit
    ): RESPONSE? = responseBodyExtractor?.validateAndExtract(
        responseHeaders,
        contentLength,
        responseBodyCapturingValidator,
        bodyFactory,
        onBodyReady
    )

    private inline fun <T> NetworkBodyExtractor<T>.validateAndExtract(
        requestHeaders: Map<String, String>,
        contentLength: Long,
        validator: NetworkLogBodyValidator?,
        bodyFactory: () -> T,
        noinline onBodyReady: (body: String?, length: Long?) -> Unit
    ): T? = validator?.invoke(requestHeaders, contentLength)?.let { (isValid, justification) ->
        if (isValid) {
            bodyFactory
                .invoke()
                .let { extract(it, onBodyReady) }
        } else {
            justification?.let { onBodyReady.invoke(it, null) }
            null
        }
    }
}
