package com.instabug.library.networkinterception.config

import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.sharedpreferences.PreferencePropertyFactory
import com.instabug.library.internal.sharedpreferences.PreferencesProperty
import com.instabug.library.percentagefeatures.PercentageFeature
import com.instabug.library.percentagefeatures.resolvePercentages
import com.instabug.library.util.hasFlag

class NetworkInterceptionConfigurationProviderImpl(
    preferencesPropertyFactory: PreferencePropertyFactory,
) : IBGNetworkInterceptionConfigurationProvider {

    private val instabugEnabled: Boolean
        get() = InstabugCore.getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED

    private val w3CNetworkExternalTraceIdPercentageFeatureDelegate =
        preferencesPropertyFactory.create<PercentageFeature?>(
            KEY_W3C_EXTERNAL_TRACE_ID_AVAILABLE,
            PercentageFeature()
        )

    private val autoMaskingPercentageFeatureDelegate =
        preferencesPropertyFactory.create<PercentageFeature?>(
            KEY_AUTO_MASKING_AVAILABLE,
            PercentageFeature().apply { beValue = DEFAULT_AUTO_MASKING_PERCENTAGE },
        )

    private var w3CNetworkExternalTraceIdPercentageFeature: PercentageFeature?
        by w3CNetworkExternalTraceIdPercentageFeatureDelegate

    private var autoMaskingPercentageFeature: PercentageFeature?
        by autoMaskingPercentageFeatureDelegate

    override fun setW3CNetworkExternalTraceIdFeatureAvailabilityPercentage(percentage: Double) {
        w3CNetworkExternalTraceIdPercentageFeature =
            w3CNetworkExternalTraceIdPercentageFeature?.apply {
                resolvePercentages(percentage)
            }
    }

    override fun isW3CNetworkExternalTraceIdEnabled(): Boolean =
        w3CNetworkExternalTraceIdPercentageFeature?.isEnabled
            ?: DEFAULT_W3C_EXTERNAL_TRACE_ID_AVAILABLE

    private val v2InterceptionPercentageFeatureDel =
        preferencesPropertyFactory.create<PercentageFeature?>(
            KEY_V2_INTERCEPTION_AVAILABLE,
            PercentageFeature().apply { beValue = DEFAULT_V2_INTERCEPTION_PERCENTAGE }
        )
    private var v2InterceptionPercentageFeature: PercentageFeature?
        by v2InterceptionPercentageFeatureDel

    override fun setV2InterceptionFeatureAvailabilityPercentage(percentage: Double) {
        v2InterceptionPercentageFeature =
            v2InterceptionPercentageFeature?.apply {
                resolvePercentages(percentage)
            }
    }

    override fun isV2InterceptionFeatureAvailable(): Boolean =
        v2InterceptionPercentageFeature?.isEnabled ?: DEFAULT_V2_INTERCEPTION_AVAILABLE

    private val isAttachingGeneratedW3CExternalTraceIdEnabledDelegate =
        preferencesPropertyFactory.create(
            KEY_GENERATED_W3C_ATTACHING_AVAILABLE,
            DEFAULT_GENERATED_W3C_ATTACHING_AVAILABLE
        )
    override var isAttachingGeneratedW3CExternalTraceIdFeatureAvailable: Boolean
        by isAttachingGeneratedW3CExternalTraceIdEnabledDelegate
    override val isAttachingGeneratedW3CExternalTraceIdEnabled: Boolean
        get() = isAttachingGeneratedW3CExternalTraceIdFeatureAvailable &&
                isW3CNetworkExternalTraceIdEnabled()

    private val isAttachingCapturedW3CExternalTraceIdEnabledDelegate =
        preferencesPropertyFactory.create(
            KEY_CAPTURED_W3C_ATTACHING_AVAILABLE,
            DEFAULT_CAPTURED_W3C_ATTACHING_AVAILABLE
        )
    override var isAttachingCapturedW3CExternalTraceIdFeatureAvailable: Boolean
        by isAttachingCapturedW3CExternalTraceIdEnabledDelegate

    override val isAttachingCapturedW3CExternalTraceIdEnabled: Boolean
        get() = isAttachingCapturedW3CExternalTraceIdFeatureAvailable &&
                isW3CNetworkExternalTraceIdEnabled()

    override val networkInterceptionEnabled: Boolean
        get() = instabugEnabled

    override val networkInterceptionV2Enabled: Boolean
        get() = networkInterceptionEnabled && isV2InterceptionFeatureAvailable()

    private val v2MaxBodySizeDelegate =
        preferencesPropertyFactory.create(
            KEY_V2_MAX_BODY_SIZE,
            DEFAULT_V2_MAX_BODY_SIZE
        )
    override var v2MaxAllowedBodySizeBytes: Long by v2MaxBodySizeDelegate

    private val v2OkHttpStateDel: PreferencesProperty<Int> =
        preferencesPropertyFactory.create(
            KEY_V2_OK_HTTP_STATE,
            DEFAULT_V2_INTERCEPTOR_STATE
        )
    override var v2OkHttpInterceptorBEState: Int by v2OkHttpStateDel

    private val v2UrlConnectionStateDel: PreferencesProperty<Int> =
        preferencesPropertyFactory.create(
            KEY_V2_URL_CONNECTION_STATE,
            DEFAULT_V2_INTERCEPTOR_STATE
        )
    override var v2UrlConnectionInterceptorBEState: Int by v2UrlConnectionStateDel

    private val v2GrpcStateDel: PreferencesProperty<Int> =
        preferencesPropertyFactory.create(
            KEY_V2_GRPC_STATE,
            DEFAULT_V2_INTERCEPTOR_STATE
        )
    override var v2GrpcInterceptorBEState: Int by v2GrpcStateDel

    private val v2GqlFeatureAvailableDel: PreferencesProperty<Boolean> =
        preferencesPropertyFactory.create(
            KEY_V2_GQL_FEATURE_AVAILABLE,
            DEFAULT_V2_GQL_FEATURE_AVAILABLE
        )
    override var v2GqlFeatureAvailable: Boolean by v2GqlFeatureAvailableDel

    private val v2ManualSanitizationAvailabilityDel: PreferencesProperty<Boolean> =
        preferencesPropertyFactory.create(
            KEY_V2_MANUAL_SANITIZATION_AVAILABLE,
            DEFAULT_V2_MANUAL_SANITIZATION_AVAILABLE
        )
    override var v2ManualSanitizationAvailable: Boolean by v2ManualSanitizationAvailabilityDel

    override val v2ManualSanitizationEnabled: Boolean
        get() = networkInterceptionV2Enabled && v2ManualSanitizationAvailable

    private val v2InstabugSanitizationQueryParamsDel: PreferencesProperty<Set<String>> =
        preferencesPropertyFactory.create(
            KEY_V2_INSTABUG_SANITIZATION_QUERY_PARAMS,
            DEFAULT_V2_INSTABUG_SANITIZATION_QUERY_PARAMS
        )
    override var v2InstabugSanitizationQueryParams: Set<String> by v2InstabugSanitizationQueryParamsDel

    override fun Int.isV2InterceptorEnabled(): Boolean =
        networkInterceptionV2Enabled && hasFlag(V2InterceptorFlags.ENABLED)

    override fun Int.isV1InterceptorEnabled(): Boolean =
        networkInterceptionEnabled &&
             !(networkInterceptionV2Enabled && hasFlag(V2InterceptorFlags.ENABLED))

    override fun setAutoMaskingFeatureAvailabilityPercentage(percentage: Double) {
        val newPercentageFeature = PercentageFeature()
        autoMaskingPercentageFeature?.also { existing -> newPercentageFeature.fromJson(existing.toJson()) }
        autoMaskingPercentageFeature = newPercentageFeature.apply { resolvePercentages(percentage) }
    }

    override fun isAutoMaskingAvailable(): Boolean =
        autoMaskingPercentageFeature?.isEnabled ?: DEFAULT_AUTO_MASKING_BE_ENABLED

    override var isAutoMaskingSDKEnabled: Boolean = DEFAULT_AUTO_MASKING_SDK_ENABLED

    override val isAutoMaskingEnabled: Boolean
        get() = isAutoMaskingAvailable() && isAutoMaskingSDKEnabled

    private val autoMaskingBEHeaderKeysDelegate =
        preferencesPropertyFactory.create(
            KEY_AUTO_MASKING_HEADER_KEYS,
            DEFAULT_SDK_AUTO_MASKING_HEADER_KEYS,
        )

    override var autoMaskingBEHeaderKeys: Set<String>
        by autoMaskingBEHeaderKeysDelegate

    private val autoMaskingBEQueryKeysDelegate =
        preferencesPropertyFactory.create(
            KEY_AUTO_MASKING_QUERY_KEYS,
            DEFAULT_SDK_AUTO_MASKING_QUERY_KEYS,
        )

    override var autoMaskingBEQueryKeys: Set<String>
        by autoMaskingBEQueryKeysDelegate

    override fun reset() {
        resetV2InterceptionFlags()
        w3CNetworkExternalTraceIdPercentageFeatureDelegate.clear()
        isAttachingGeneratedW3CExternalTraceIdEnabledDelegate.clear()
        isAttachingCapturedW3CExternalTraceIdEnabledDelegate.clear()
        networkBodyCaptureRuleDelegation.clear()
        resetAutoMasking()
    }

    override fun resetAutoMasking() {
        autoMaskingPercentageFeatureDelegate.clear()
        autoMaskingBEQueryKeysDelegate.clear()
        autoMaskingBEHeaderKeysDelegate.clear()
    }

    override fun resetV2InterceptionFlags() {
        resetV2SanitizationFlags()
        v2InterceptionPercentageFeatureDel.clear()
        v2OkHttpStateDel.clear()
        v2UrlConnectionStateDel.clear()
        v2GrpcStateDel.clear()
        v2GqlFeatureAvailableDel.clear()
    }

    override fun resetV2SanitizationFlags() {
        v2ManualSanitizationAvailabilityDel.clear()
        v2InstabugSanitizationQueryParamsDel.clear()
    }

    private val networkBodyCaptureRuleDelegation = preferencesPropertyFactory.create(
        KEY_BE_CAPTURE_NETWORK_BODY_LOGS,
        DEFAULT_CAPTURE_NETWORK_BODY_LOG_BE
    )

    override var isNetworkLogBodyEnabledByBE: Boolean
        by networkBodyCaptureRuleDelegation
    override var isNetworkLogBodyEnabledBySDK: Boolean = DEFAULT_CAPTURE_NETWORK_BODY_LOG_SDK
    override val shouldCaptureNetworkBodyLogs: Boolean
        get() =
            isNetworkLogBodyEnabledBySDK && isNetworkLogBodyEnabledByBE
}
