package com.instabug.library.networkinterception

import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsHandler
import com.instabug.library.fill.Filler
import com.instabug.library.internal.crossplatform.CPConfigurationsProvider
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.internal.servicelocator.Provider
import com.instabug.library.internal.sharedpreferences.corePrefFactory
import com.instabug.library.logging.listeners.networklogs.NetworkLogListener
import com.instabug.library.logging.listeners.networklogs.NetworkLogSnapshot
import com.instabug.library.map.Mapper
import com.instabug.library.model.NetworkLog
import com.instabug.library.networkinterception.config.DEFAULT_SDK_AUTO_MASKING_KEYS
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.config.NetworkInterceptionConfigurationHandler
import com.instabug.library.networkinterception.config.NetworkInterceptionConfigurationProviderImpl
import com.instabug.library.networkinterception.delegate.DefaultNetworkInterceptorDelegate
import com.instabug.library.networkinterception.delegate.NetworkInterceptorDelegate
import com.instabug.library.networkinterception.delegate.Utf8CodePointEvaluator
import com.instabug.library.networkinterception.delegate.extract.ImmediateInputStreamExtractor
import com.instabug.library.networkinterception.delegate.extract.InputStreamWrapperFactory
import com.instabug.library.networkinterception.delegate.extract.LazyInputStreamExtractor
import com.instabug.library.networkinterception.delegate.extract.LazyOutputStreamExtractor
import com.instabug.library.networkinterception.delegate.extract.NetworkBodyExtractor
import com.instabug.library.networkinterception.delegate.extract.OutputStreamWrapperFactory
import com.instabug.library.networkinterception.delegate.validate.ByteArrayUtf8PlainTextValidator
import com.instabug.library.networkinterception.delegate.validate.DefaultRequestBodyValidator
import com.instabug.library.networkinterception.delegate.validate.DefaultResponseBodyValidator
import com.instabug.library.networkinterception.delegate.validate.NetworkLogBodyValidator
import com.instabug.library.networkinterception.dispatch.NetworkLogDispatcher
import com.instabug.library.networkinterception.dispatch.NetworkLogReceiverRegistry
import com.instabug.library.networkinterception.dispatch.SynchronousNetworkLogDispatcher
import com.instabug.library.networkinterception.externaltrace.RandomUIntProvider
import com.instabug.library.networkinterception.externaltrace.TimestampProviderSeconds
import com.instabug.library.networkinterception.externaltrace.W3CExternalHeaderSanitizer
import com.instabug.library.networkinterception.externaltrace.W3CExternalNetworkTraceIdFactory
import com.instabug.library.networkinterception.externaltrace.W3CFormatProvider
import com.instabug.library.networkinterception.externaltrace.W3CHeadersInvalidator
import com.instabug.library.networkinterception.gql.GqlHeaderToQueryMapper
import com.instabug.library.networkinterception.invalidate.CompositeHeadersInvalidator
import com.instabug.library.networkinterception.invalidate.HeadersInvalidator
import com.instabug.library.networkinterception.model.NetworkLogModel
import com.instabug.library.networkinterception.model.NetworkLogW3CExternalTraceIdInfo
import com.instabug.library.networkinterception.model.map.CoreCommonNetworkLogJsonMapper
import com.instabug.library.networkinterception.model.map.NetworkLogBuilderToSnapshotMapper
import com.instabug.library.networkinterception.model.map.NetworkSnapshotToNetworkLogBuilderFiller
import com.instabug.library.networkinterception.okhttp.delegate.OkHttpInterceptorDelegateProvider
import com.instabug.library.networkinterception.okhttp.delegate.validate.OkHttpRequestBodyValidator
import com.instabug.library.networkinterception.okhttp.delegate.validate.OkHttpResponseBodyValidator
import com.instabug.library.networkinterception.sanitization.AutoMaskingSanitizer
import com.instabug.library.networkinterception.sanitization.AutoMaskingSanitizerV2
import com.instabug.library.networkinterception.sanitization.BodySizeSanitizer
import com.instabug.library.networkinterception.sanitization.HandleRequestAndResponseBodySanitizer
import com.instabug.library.networkinterception.sanitization.InstabugQueryParamsSanitizer
import com.instabug.library.networkinterception.sanitization.UserDefinedSanitizer
import com.instabug.library.networkinterception.urlconnection.IBGHttpUrlConnection
import com.instabug.library.networkinterception.urlconnection.IBGHttpUrlConnectionFactory
import com.instabug.library.networkinterception.urlconnection.IBGHttpsUrlConnection
import com.instabug.library.networkinterception.urlconnection.IBGHttpsUrlConnectionFactory
import com.instabug.library.networkinterception.urlconnection.delegate.UrlConnectionInterceptorDelegateProvider
import com.instabug.library.networkinterception.urlconnection.delegate.validate.UrlConnectionRequestBodyValidator
import com.instabug.library.networkinterception.urlconnection.delegate.validate.UrlConnectionResponseBodyValidator
import com.instabug.library.networkv2.utils.IBGDomainProvider
import com.instabug.library.sanitize.CompositeSanitizer
import com.instabug.library.sanitize.Sanitizer
import com.instabug.library.sanitize.Validator
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator
import com.instabug.library.util.threading.OrderedExecutor
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.weakRetryLazy
import org.json.JSONObject
import java.io.InputStream
import java.io.OutputStream
import java.net.HttpURLConnection
import java.net.URL
import java.util.concurrent.ExecutorService
import javax.net.ssl.HttpsURLConnection

typealias W3CHeaderInfoFactory =
    ParameterizedFactory<NetworkLogW3CExternalTraceIdInfo?, Map<String, String>?>

object NetworkInterceptionServiceLocator {

    private const val EXECUTOR_KEY = "V2NetworkInterception"
    private const val BODY_SAMPLE_SIZE = 64

    @JvmStatic
    val configurationProvider: IBGNetworkInterceptionConfigurationProvider by lazy {
        NetworkInterceptionConfigurationProviderImpl(corePrefFactory())
    }

    @JvmStatic
    val configurationHandler: FeatureFlagsConfigsHandler
        get() = NetworkInterceptionConfigurationHandler(
            configurationProvider,
            CPConfigurationsProvider
        )

    @JvmStatic
    private val synchronousNetworkLogDispatcher: SynchronousNetworkLogDispatcher by lazy {
        SynchronousNetworkLogDispatcher().apply {
            addReceiver(CoreServiceLocator.dataHubNetworkLogReceiver)
            addReceiver(SessionReplayServiceLocator.networkLogReceiver)
        }
    }

    @JvmStatic
    val networkLogDispatcher: NetworkLogDispatcher
        get() = synchronousNetworkLogDispatcher

    @JvmStatic
    val networkLogReceiverRegistry: NetworkLogReceiverRegistry
        get() = synchronousNetworkLogDispatcher

    @JvmStatic
    val bucket: NetworkInterceptionBucket by lazy {
        NetworkInterceptionBucketImpl()
    }

    @JvmStatic
    val apiProxy: NetworkInterceptionApiProxy
        get() = NetworkInterceptionApiProxy(bucket)

    @JvmStatic
    val manualSanitizerListenerProvider: Provider<NetworkLogListener?>
        get() = Provider { bucket.networkLogListener }

    private val networkLogBuilderToSnapshotMapper: Mapper<NetworkLogModel.Builder, NetworkLogSnapshot>
        get() = NetworkLogBuilderToSnapshotMapper()

    private val networkSnapshotToNetworkLogBuilderFiller: Filler<NetworkLogModel.Builder, NetworkLogSnapshot>
        get() = NetworkSnapshotToNetworkLogBuilderFiller()

    private inline val userDefinedSanitizer: Sanitizer<NetworkLogModel.Builder>
        get() = UserDefinedSanitizer(
            manualSanitizerListenerProvider,
            networkLogBuilderToSnapshotMapper,
            networkSnapshotToNetworkLogBuilderFiller,
            configurationProvider
        )

    private inline val instabugQueryParamsSanitizer: Sanitizer<NetworkLogModel.Builder>
        get() = InstabugQueryParamsSanitizer(
            setOf(
                IBGDomainProvider.getInstabugDomain(),
                IBGDomainProvider.getAPMDomain()
            ),
            configurationProvider
        )

    private inline val autoMaskingSanitizerV2: Sanitizer<NetworkLogModel.Builder>
        get() = AutoMaskingSanitizerV2(
            autoMaskingHeaderKeysLowerCase,
            autoMaskingQueryKeysLowerCase,
            configurationProvider
        )
    private inline val handleRequestAndResponseBodySanitizer: Sanitizer<NetworkLogModel.Builder>
        get() = HandleRequestAndResponseBodySanitizer(
            configurationProvider
        )

    private inline val w3cHeaderInfoFactory: W3CHeaderInfoFactory
        get() = W3CExternalNetworkTraceIdFactory(
            configurationProvider,
            W3CFormatProvider(
                TimestampProviderSeconds(),
                RandomUIntProvider()
            )
        )

    private inline val w3CHeadersInfoSanitizer: Sanitizer<NetworkLogModel.Builder>
        get() = W3CExternalHeaderSanitizer(
            configurationProvider,
            w3cHeaderInfoFactory
        )

    private inline val bodySizeSanitizer
        get() = BodySizeSanitizer(configurationProvider)

    private val sanitizer: Sanitizer<NetworkLogModel.Builder>
        get() = CompositeSanitizer<NetworkLogModel.Builder>().apply {
            addSanitizer(instabugQueryParamsSanitizer)
            addSanitizer(w3CHeadersInfoSanitizer)
            addSanitizer(autoMaskingSanitizerV2)
            addSanitizer(bodySizeSanitizer)
            addSanitizer(userDefinedSanitizer)
            addSanitizer(handleRequestAndResponseBodySanitizer)
        }

    private val executor: ExecutorService
        get() = OrderedExecutor(EXECUTOR_KEY, PoolProvider.getInstance().orderedExecutor)

    val processor: NetworkLogProcessor by weakRetryLazy {
        NetworkLogProcessorImpl(
            configurationProvider,
            sanitizer,
            networkLogDispatcher,
            executor
        )
    }

    private val gqlHeaderToQueryMapper: Mapper<Map<String, String>, String?>
        get() = GqlHeaderToQueryMapper(configurationProvider)

    private val w3cHeadersInvalidator: HeadersInvalidator
        get() = W3CHeadersInvalidator(w3cHeaderInfoFactory)

    private val headersInvalidator: HeadersInvalidator by weakRetryLazy {
        CompositeHeadersInvalidator(
            listOf(w3cHeadersInvalidator)
        )
    }

    private val utf8CodePointEvaluator: Utf8CodePointEvaluator
        get() = Utf8CodePointEvaluator()

    private val byteArrayUtf8PlainTextValidator: Validator<ByteArray>
        get() = ByteArrayUtf8PlainTextValidator(utf8CodePointEvaluator)

    private val immediateInputStreamExtractor: NetworkBodyExtractor<InputStream>
        get() = ImmediateInputStreamExtractor(
            BODY_SAMPLE_SIZE,
            configurationProvider,
            byteArrayUtf8PlainTextValidator
        )

    private val lazyOutputStreamExtractor: NetworkBodyExtractor<OutputStream>
        get() = LazyOutputStreamExtractor(
            OutputStreamWrapperFactory(
                configurationProvider,
                BODY_SAMPLE_SIZE,
                utf8CodePointEvaluator
            )
        )

    private val lazyInputStreamExtractor: NetworkBodyExtractor<InputStream>
        get() = LazyInputStreamExtractor(
            InputStreamWrapperFactory(
                configurationProvider,
                BODY_SAMPLE_SIZE,
                utf8CodePointEvaluator
            )
        )

    private val defaultRequestBodyValidator: NetworkLogBodyValidator
        get() = DefaultRequestBodyValidator(configurationProvider)

    private val defaultResponseBodyValidator: NetworkLogBodyValidator by weakRetryLazy {
        DefaultResponseBodyValidator(configurationProvider)
    }

    private val okHttpRequestBodyCapturingValidator: NetworkLogBodyValidator
        get() = OkHttpRequestBodyValidator(defaultRequestBodyValidator, configurationProvider)

    private val okHttpResponseBodyCapturingValidator: NetworkLogBodyValidator
        get() = OkHttpResponseBodyValidator(configurationProvider)

    val okHttpInterceptorDelegate: NetworkInterceptorDelegate<InputStream, InputStream> by weakRetryLazy {
        val bodyExtractor: NetworkBodyExtractor<InputStream> = immediateInputStreamExtractor
        DefaultNetworkInterceptorDelegate(
            graphQlQueryMapper = gqlHeaderToQueryMapper,
            requestHeadersInvalidators = headersInvalidator,
            requestBodyCapturingValidator = okHttpRequestBodyCapturingValidator,
            responseBodyCapturingValidator = okHttpResponseBodyCapturingValidator,
            requestBodyExtractor = bodyExtractor,
            responseBodyExtractor = bodyExtractor
        )
    }

    val okHttpInterceptorDelegateProvider: Provider<NetworkInterceptorDelegate<InputStream, InputStream>?>
        get() = OkHttpInterceptorDelegateProvider()

    val urlConnectionInterceptorDelegate: NetworkInterceptorDelegate<OutputStream, InputStream> by weakRetryLazy {
        DefaultNetworkInterceptorDelegate(
            graphQlQueryMapper = null,
            requestHeadersInvalidators = headersInvalidator,
            requestBodyCapturingValidator = UrlConnectionRequestBodyValidator(
                defaultRequestBodyValidator,
                configurationProvider
            ),
            responseBodyCapturingValidator = UrlConnectionResponseBodyValidator(
                defaultResponseBodyValidator,
                configurationProvider
            ),
            requestBodyExtractor = lazyOutputStreamExtractor,
            responseBodyExtractor = lazyInputStreamExtractor,
        )
    }

    private val urlConnectionInterceptorDelegateProvider: Provider<NetworkInterceptorDelegate<OutputStream, InputStream>?>
        get() = UrlConnectionInterceptorDelegateProvider()

    val ibgHttpUrlConnectionFactory: ParameterizedFactory<IBGHttpUrlConnection?, Pair<URL, HttpURLConnection>>
    by weakRetryLazy { IBGHttpUrlConnectionFactory(urlConnectionInterceptorDelegateProvider) }

    val ibgHttpsUrlConnectionFactory: ParameterizedFactory<IBGHttpsUrlConnection?, Pair<URL, HttpsURLConnection>>
    by weakRetryLazy { IBGHttpsUrlConnectionFactory(urlConnectionInterceptorDelegateProvider) }

    @JvmStatic
    fun getAutoMaskingSanitizer(): Sanitizer<NetworkLog> =
        AutoMaskingSanitizer(
            autoMaskingHeaderKeysLowerCase,
            autoMaskingQueryKeysLowerCase,
        )

    private val autoMaskingHeaderKeysLowerCase: Set<String>
        get() {
            val keysSet = mutableSetOf<String>()
            DEFAULT_SDK_AUTO_MASKING_KEYS.forEach {
                keysSet.add(it.lowercase())
            }
            configurationProvider.autoMaskingBEHeaderKeys.forEach {
                keysSet.add(it.lowercase())
            }
            return keysSet.toSet()
        }

    private val autoMaskingQueryKeysLowerCase: Set<String>
        get() {
            val keysSet = mutableSetOf<String>()
            DEFAULT_SDK_AUTO_MASKING_KEYS.forEach {
                keysSet.add(it.lowercase())
            }
            configurationProvider.autoMaskingBEQueryKeys.forEach {
                keysSet.add(it.lowercase())
            }
            return keysSet
        }

    @JvmStatic
    val coreCommonNetworkLogJsonMapper: Mapper<NetworkLogModel, JSONObject?>
        get() = CoreCommonNetworkLogJsonMapper()
}
