package com.instabug.library.internal.storage.cache.db;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementAssetsEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.InstabugLogEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SDKApiEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SDKEventEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEntity;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEventEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions;

import android.annotation.SuppressLint;
import android.content.Context;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.os.Build;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.internal.storage.cache.db.migrations.DestructiveMigration;
import com.instabug.library.logging.InstabugLog;

/**
 * Created by mNagy on 12/19/16.
 */

public class InstabugDbHelper extends SQLiteOpenHelper {

    private static final String DATABASE_NAME = "instabug.db";
    @Nullable
    private static InstabugDbHelper instance;

    public synchronized static SQLiteOpenHelper getInstance(@Nullable Context context) {
        if (instance == null) {
            instance = new InstabugDbHelper(context);
        }
        return instance;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private InstabugDbHelper(@Nullable Context context) {
        super(context, DATABASE_NAME, null, InstabugDBVersions.CURRENT_VERSION);
    }

    @Override
    public void onConfigure(SQLiteDatabase db) {
        super.onConfigure(db);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            db.setForeignKeyConstraintsEnabled(true);
        }
    }

    @Override
    public void onCreate(SQLiteDatabase sqLiteDatabase) {
        new DestructiveMigration().createTables(sqLiteDatabase);
    }

    @Override
    @SuppressLint("STRICT_MODE_VIOLATION")
    public void onUpgrade(final SQLiteDatabase db, final int oldVersion, final int newVersion) {
        updateTablesOnSchemaChanges(db, oldVersion, newVersion);
        MigrationEngine.onUpgrade(db, oldVersion, newVersion);
        onCreate(db);
    }

    @Override
    @SuppressLint("STRICT_MODE_VIOLATION")
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        updateTablesOnSchemaChanges(db, oldVersion, newVersion);
        MigrationEngine.onDowngrade(db, oldVersion, newVersion);
        onCreate(db);
    }

    @Override
    public void onOpen(SQLiteDatabase db) {
        super.onOpen(db);
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
            db.execSQL("PRAGMA foreign_keys=ON;");
        }
    }

    /**
     * If your migration process behave unexpectedly and the target table is dropped, then remove
     * the query that drops your table from this method [Da if you found it b2a immediately :'(].
     * <p>
     * We are removing them one by one when an unexpected behaviour is occurred everytime the
     * database was changed or migration was applied.
     */
    @Deprecated
    private void updateTablesOnSchemaChanges(SQLiteDatabase sqLiteDatabase, int oldVersion, int newVersion) {
        handleSurveysMigration(sqLiteDatabase, oldVersion, newVersion);
        sqLiteDatabase.execSQL(InstabugLogEntry.DROP_QUERY_INSTABUG_LOG);
        sqLiteDatabase.execSQL(UserEventEntry.DROP_TABLE);
        sqLiteDatabase.execSQL(SDKEventEntry.DROP_TABLE);
        sqLiteDatabase.execSQL(SDKApiEntry.DROP_TABLE);
        sqLiteDatabase.execSQL(FeatureRequestEntry.DROP_TABLE);
        onCreate(sqLiteDatabase);
    }

    /**
     * Starting from SDK ver. 8.4.0 (DB ver. 12), UserInteractions on surveys was introduced
     * Any further update in Instabug DB schema should migrate and keep all data related to surveys and interactions
     * In previous versions we drop/recreate tables
     * {@since 8.6.0}
     */
    @Deprecated
    private void handleSurveysMigration(SQLiteDatabase sqLiteDatabase, int oldVersion, int newVersion) {
        if (newVersion > oldVersion) {
            // upgrading
            if (newVersion > InstabugDBVersions.DATABASE_VERSION_16) {
                if (oldVersion <= InstabugDBVersions.DATABASE_VERSION_16) {
                    try {
                        sqLiteDatabase.execSQL(AnnouncementEntry.ALTER_TABLE_UPGRADE_16);
                    } catch (SQLException e) {
                        dropAnnouncementTables(sqLiteDatabase);
                        InstabugLog.e("Migration of schema v. " + InstabugDBVersions.DATABASE_VERSION_16 +
                                " failed with the error: " + e.getMessage());
                    }
                }
            }

            if (newVersion >= InstabugDBVersions.DATABASE_VERSION_15) {
                if (oldVersion < InstabugDBVersions.DATABASE_VERSION_12) {
                    // We drop all tables from versions below 8.4.0 since there was no archiving yet.
                    dropSurveysTables(sqLiteDatabase);
                    dropAnnouncementTables(sqLiteDatabase);
                } else if (oldVersion == InstabugDBVersions.DATABASE_VERSION_12) {
                    try {
                        sqLiteDatabase.execSQL(UserEntity.ALTER_TABLE_UPGRADE_12);
                        sqLiteDatabase.execSQL(UserAttributesEntry.ALTER_TABLE_UPGRADE_14);
                    } catch (SQLException e) {
                        dropSurveysTables(sqLiteDatabase);
                        dropAnnouncementTables(sqLiteDatabase);
                        InstabugLog.e("Migration of schema v. " + InstabugDBVersions.DATABASE_VERSION_12 +
                                " failed with the error: " + e.getMessage());
                    }
                } else if (oldVersion == InstabugDBVersions.DATABASE_VERSION_14) {
                    try {
                        sqLiteDatabase.execSQL(UserAttributesEntry.ALTER_TABLE_UPGRADE_14);
                    } catch (SQLException e) {
                        dropSurveysTables(sqLiteDatabase);
                        dropAnnouncementTables(sqLiteDatabase);
                        InstabugLog.e("Migration of schema v. " + InstabugDBVersions.DATABASE_VERSION_14 +
                                " failed with the error: " + e.getMessage());
                    }
                }
            }
        } else if (newVersion < oldVersion) {
            // downgrading
        }
    }

    private void dropSurveysTables(SQLiteDatabase sqLiteDatabase) {
        sqLiteDatabase.execSQL(SurveyEntry.DROP_TABLE);
        sqLiteDatabase.execSQL(UserInteractions.DROP_TABLE);
        sqLiteDatabase.execSQL(UserAttributesEntry.DROP_TABLE);
        sqLiteDatabase.execSQL(UserEntity.DROP_TABLE);
    }

    private void dropAnnouncementTables(SQLiteDatabase sqLiteDatabase) {
        sqLiteDatabase.execSQL(AnnouncementEntry.DROP_TABLE);
        sqLiteDatabase.execSQL(AnnouncementAssetsEntry.DROP_TABLE);
    }

    @VisibleForTesting
    public static void clearInstance() {
        instance = null;
    }

}
