package com.instabug.library.internal.resolver;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.model.LoggingSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * A logging setting resolver which will hold the logging setting to get any settings
 * Also, it start an observer to listen on logging level changes.
 */
public class LoggingSettingResolver {

    private static LoggingSettingResolver loggingSettingResolver;

    private static final String TAG = "LoggingSettingResolver";
    @Nullable
    private volatile LoggingSettings loggingSettings;
    private boolean lastEncrypted;

    private LoggingSettingResolver() {
        getLoggingSettingsFromPreferences();
        updateLastEncrypted();
    }

    /**
     * update the lastEncrypted Param that loaded from the disk
     * or after updating that the encryption change to continue deleting logic
     */
    private void updateLastEncrypted() {
        LoggingSettings internalLoggingSetting = loggingSettings;
        lastEncrypted = internalLoggingSetting != null && internalLoggingSetting.isEncrypted();
    }

    public static synchronized LoggingSettingResolver getInstance() {
        if (loggingSettingResolver == null) {
            loggingSettingResolver = new LoggingSettingResolver();
        }
        return loggingSettingResolver;
    }


    /**
     * Resolve logging settings from json object to return a logging setting
     * it will return a new object with the new settings.
     *
     * @param jsonObject response of logging settings.
     * @return logging settings.
     * @throws JSONException
     */
    private LoggingSettings resolveLoggingSettings(@NonNull JSONObject jsonObject) throws JSONException {
        LoggingSettings loggingSettings = new LoggingSettings();
        loggingSettings.fromJson(jsonObject);
        return loggingSettings;
    }

    /**
     * * Sets the logging feature settings from the provided JSON object.
     * <p>
     * If the input JSON is {@code null}, this will reset the internal {@link LoggingSettings}
     * * to its default state, clear any previously persisted settings, and notify observers
     * * of the state change with {@code null}.
     * </p>
     * <p>
     * If a valid JSON object is provided, it will be parsed and converted into a
     * {@link LoggingSettings} instance. The parsed settings will be persisted using the
     * {@link SettingsManager} and any registered observers will be notified of the
     * updated state.
     * </p>
     *
     * @param sdkLogFeatureJSon {@link JSONObject}
     * @throws JSONException
     */
    public void setLoggingFeatureSettings(@Nullable JSONObject sdkLogFeatureJSon) throws JSONException {
        if (sdkLogFeatureJSon == null) {
            loggingSettings = new LoggingSettings();
            notifyStateChange(loggingSettings);
            SettingsManager.getInstance().setLoggingFeatureSettings(null);
            return;
        }
        this.loggingSettings = resolveLoggingSettings(sdkLogFeatureJSon);
        SettingsManager.getInstance().setLoggingFeatureSettings(sdkLogFeatureJSon.toString());
        if (loggingSettings != null)
            notifyStateChange(loggingSettings);
    }

    private void notifyStateChange(@Nullable LoggingSettings loggingSettings) {
        if (loggingSettings != null) {
            markIfEncryptionChanged(loggingSettings);
            InstabugSDKLogger.onDiskLoggingSettingsChanged(loggingSettings);
            updateLastEncrypted();
        }
    }

    private void markIfEncryptionChanged(@Nullable LoggingSettings loggingSettings) {
        if (loggingSettings != null) {
            boolean isEncryptionFlagChanged = lastEncrypted != loggingSettings.isEncrypted();
            loggingSettings.setShouldClearDiskLogs(isEncryptionFlagChanged);
        }
    }

    /**
     * A method to get logging setting
     *
     * @return logging setting or null if logging setting not found which means that the logging should be disabled
     */
    @Nullable
    public LoggingSettings getLoggingSettings() {
        return loggingSettings;
    }

    private void getLoggingSettingsFromPreferences() {
        try {
            if (loggingSettings == null) {
                String json = SettingsManager.getInstance().getLoggingFeatureSettings();
                if (json != null) {
                    LoggingSettings newLoggingSettings = new LoggingSettings();
                    newLoggingSettings.fromJson(json);
                    loggingSettings = newLoggingSettings;
                }
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.toString(), e);
        }
    }
}
