package com.instabug.library.featuresflags.configs

import androidx.annotation.VisibleForTesting
import com.instabug.library.featuresflags.constants.FFMode
import com.instabug.library.featuresflags.managers.FeatureFlagsManager
import com.instabug.library.internal.sharedpreferences.corePref
import org.json.JSONObject

@VisibleForTesting
const val KEY_FEATURES_FLAGS = "feature_flags"

@VisibleForTesting
const val KEY_FEATURES_FLAGS_MODE = "mode"

@VisibleForTesting
const val KEY_FEATURES_FLAGS_STORE_LIMIT = "limit"

@VisibleForTesting
const val FEATURE_FEATURES_FLAGS_MODE_DEFAULT = FFMode.EXPERIMENTS

@VisibleForTesting
const val FEATURE_FEATURES_FLAGS_STORE_LIMIT_DEFAULT = 200

fun interface FeatureFlagsConfigsHandler {
    fun handleConfigs(featuresResponse: JSONObject)
    fun reset() {
        // Optional
    }
}

/**
 * A contract that provides the Features Flags feature configurations.
 */
interface FeatureFlagsConfigsProvider {
    /**
     * Feature mode that's controlling the feature availability and behaviour,
     * it fails back to {#FEATURE_FEATURES_FLAGS_MODE_DEFAULT} as a default.
     */
    @FFMode
    var mode: Int

    /**
     * Feature storeLimit that controls the max number of features flags the can be stored,
     * it fails back to {#FEATURE_FEATURES_FLAGS_STORE_LIMIT_DEFAULT} as a default.
     */
    var storeLimit: Int
}

internal class FeatureFlagsConfigsProviderImpl : FeatureFlagsConfigsProvider {

    @FFMode
    override var mode: Int by corePref(
        KEY_FEATURES_FLAGS_MODE, FEATURE_FEATURES_FLAGS_MODE_DEFAULT
    )
    override var storeLimit: Int by corePref(
        KEY_FEATURES_FLAGS_STORE_LIMIT, FEATURE_FEATURES_FLAGS_STORE_LIMIT_DEFAULT
    )
}

internal class FeatureFlagsConfigsHandlerImpl(
    private val featureFlagsConfigsProvider: FeatureFlagsConfigsProvider,
    private val featureFlagsManager: FeatureFlagsManager
) :
    FeatureFlagsConfigsHandler {

    override fun handleConfigs(featuresResponse: JSONObject) {
        val featuresFlagsResponse = featuresResponse.optJSONObject(
            KEY_FEATURES_FLAGS
        )
        featureFlagsConfigsProvider.mode = featuresFlagsResponse?.optInt(
            KEY_FEATURES_FLAGS_MODE, FEATURE_FEATURES_FLAGS_MODE_DEFAULT
        ) ?: FEATURE_FEATURES_FLAGS_MODE_DEFAULT
        featureFlagsConfigsProvider.storeLimit = featuresFlagsResponse?.optInt(
            KEY_FEATURES_FLAGS_STORE_LIMIT, FEATURE_FEATURES_FLAGS_STORE_LIMIT_DEFAULT
        ) ?: FEATURE_FEATURES_FLAGS_STORE_LIMIT_DEFAULT

        featureFlagsConfigsProvider.mode.takeIf { it == FFMode.DISABLED }
            ?.run { featureFlagsManager.removeAllFeaturesFlags() } ?: featureFlagsManager.trim()
    }


}